/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.domain;

import static org.mule.runtime.api.util.collection.SmallMap.of;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.core.internal.lifecycle.phases.NotInLifecyclePhase;
import org.mule.runtime.deployment.model.api.domain.DomainStatus;

import java.util.Map;

/**
 * Maps a domain life cycle phase to a domain status
 */
public class DomainStatusMapper {

  private static final Map<String, DomainStatus> statusMapping = getStatusMapping();

  public static DomainStatus getDomainStatus(String currentPhase) {
    final DomainStatus domainStatus = statusMapping.get(currentPhase);

    if (domainStatus == null) {
      throw new IllegalStateException("Unknown lifecycle phase: " + currentPhase);
    }

    return domainStatus;
  }

  private static Map<String, DomainStatus> getStatusMapping() {
    Map<String, DomainStatus> statusMapping = of(
                                                 NotInLifecyclePhase.PHASE_NAME, DomainStatus.CREATED,
                                                 Disposable.PHASE_NAME, DomainStatus.DESTROYED,
                                                 Stoppable.PHASE_NAME, DomainStatus.STOPPED,
                                                 Startable.PHASE_NAME, DomainStatus.STARTED,
                                                 Initialisable.PHASE_NAME, DomainStatus.INITIALISED);

    return statusMapping;
  }
}

