/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.policy;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.api.service.ServiceRepository;
import org.mule.runtime.core.api.policy.PolicyParametrization;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.artifact.ArtifactConfigurationProcessor;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplate;
import org.mule.runtime.module.artifact.activation.api.extension.discovery.ExtensionModelLoaderRepository;
import org.mule.runtime.module.artifact.api.classloader.ClassLoaderRepository;

/**
 * Creates instances of {@link DefaultApplicationPolicyInstance}
 */
public class DefaultPolicyInstanceProviderFactory implements PolicyInstanceProviderFactory {

  private final ServiceRepository serviceRepository;
  private final ClassLoaderRepository classLoaderRepository;
  private final ExtensionModelLoaderRepository extensionModelLoaderRepository;
  private final ArtifactConfigurationProcessor artifactConfigurationProcessor;

  /**
   * Creates a new factory
   *
   * @param serviceRepository              contains available service instances. Non null.
   * @param classLoaderRepository          contains the registered classloaders that can be used to load serialized classes. Non
   *                                       null.
   * @param extensionModelLoaderRepository {@link ExtensionModelLoaderRepository} with the available extension loaders. Non null.
   * @param artifactConfigurationProcessor the processor to use for building the application model. Non null.
   */
  public DefaultPolicyInstanceProviderFactory(ServiceRepository serviceRepository,
                                              ClassLoaderRepository classLoaderRepository,
                                              ExtensionModelLoaderRepository extensionModelLoaderRepository,
                                              ArtifactConfigurationProcessor artifactConfigurationProcessor) {
    this.extensionModelLoaderRepository = extensionModelLoaderRepository;
    requireNonNull(serviceRepository, "serviceRepository cannot be null");
    requireNonNull(classLoaderRepository, "classLoaderRepository cannot be null");
    requireNonNull(extensionModelLoaderRepository, "extensionModelLoaderRepository cannot be null");
    requireNonNull(artifactConfigurationProcessor, "artifactConfigurationProcessor cannot be null");

    this.serviceRepository = serviceRepository;
    this.classLoaderRepository = classLoaderRepository;
    this.artifactConfigurationProcessor = artifactConfigurationProcessor;
  }

  @Override
  public ApplicationPolicyInstance create(Application application, PolicyTemplate policyTemplate,
                                          PolicyParametrization parametrization) {
    return new DefaultApplicationPolicyInstance(application, policyTemplate, parametrization, serviceRepository,
                                                classLoaderRepository,
                                                extensionModelLoaderRepository, null,
                                                artifactConfigurationProcessor);
  }

}
