/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.util.Optional;

import org.junit.jupiter.api.Test;

@SmallTest
public class DefaultArtifactClassLoaderManagerTestCase extends AbstractMuleTestCase {

  public static final String ARTIFACT_ID = "ID";

  private final DefaultClassLoaderManager manager = new DefaultClassLoaderManager();

  @Test
  void registersArtifactClassLoader() throws Exception {
    ArtifactClassLoader artifactClassLoader = getArtifactClassLoader();

    manager.register(artifactClassLoader);

    assertThat(manager.find(ARTIFACT_ID).get(), is(artifactClassLoader.getClassLoader()));
  }

  @Test
  void unregistersArtifactClassLoader() throws Exception {
    ArtifactClassLoader artifactClassLoader = getArtifactClassLoader();

    manager.register(artifactClassLoader);

    assertThat(manager.unregister(ARTIFACT_ID), is(artifactClassLoader));
  }

  @Test
  void getIdForRegisteredArtifactClassLoader() {
    ArtifactClassLoader artifactClassLoader = getArtifactClassLoader();

    manager.register(artifactClassLoader);

    assertThat(manager.getId(artifactClassLoader.getClassLoader()), is(of(ARTIFACT_ID)));
  }

  @Test
  void getIdForNonRegisteredArtifactClassLoader() {
    ArtifactClassLoader artifactClassLoader = getArtifactClassLoader();

    assertThat(manager.getId(artifactClassLoader.getClassLoader()), is(empty()));
  }

  @Test
  void failsToRegisterNullArtifactClassLoader() throws Exception {
    assertThrows(NullPointerException.class, () -> manager.register(null));
  }

  @Test
  void failsToUnregisterNullArtifactId() throws Exception {
    assertThrows(NullPointerException.class, () -> manager.unregister(null));
  }

  @Test
  void failsToUnregisterEmptyArtifactId() throws Exception {
    assertThrows(IllegalArgumentException.class, () -> manager.unregister(""));
  }

  @Test
  void findReturnsEmptyWhenClassloaderMissing() {
    Optional<ClassLoader> result = manager.find(ARTIFACT_ID);

    assertThat(result, is(Optional.empty()));
  }

  private ArtifactClassLoader getArtifactClassLoader() {
    ArtifactClassLoader artifactClassLoader = mock(ArtifactClassLoader.class);
    when(artifactClassLoader.getArtifactId()).thenReturn(ARTIFACT_ID);
    ClassLoader expectedClassLoader = getClass().getClassLoader();
    when(artifactClassLoader.getClassLoader()).thenReturn(expectedClassLoader);
    return artifactClassLoader;
  }
}
