/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static org.mule.runtime.api.util.Preconditions.checkArgument;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoaderManager;
import org.mule.runtime.module.artifact.api.classloader.ClassLoaderRepository;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Tracks all the {@link ArtifactClassLoader} created on the container.
 */
public class DefaultClassLoaderManager implements ArtifactClassLoaderManager, ClassLoaderRepository {

  private final Map<String, ArtifactClassLoader> artifactClassLoaders = new ConcurrentHashMap<>();

  @Override
  public void register(ArtifactClassLoader artifactClassLoader) {
    requireNonNull(artifactClassLoader, "artifactClassLoader cannot be null");

    artifactClassLoaders.put(artifactClassLoader.getArtifactId(), artifactClassLoader);
  }

  @Override
  public ArtifactClassLoader unregister(String classLoaderId) {
    checkClassLoaderId(classLoaderId);

    return artifactClassLoaders.remove(classLoaderId);
  }

  @Override
  public Optional<ClassLoader> find(String classLoaderId) {
    checkClassLoaderId(classLoaderId);

    ArtifactClassLoader artifactClassLoader = artifactClassLoaders.get(classLoaderId);

    return artifactClassLoader != null ? of(artifactClassLoader.getClassLoader()) : empty();
  }

  @Override
  public Optional<String> getId(ClassLoader classLoader) {
    return artifactClassLoaders.values().stream()
        .filter(artifactClassLoader -> artifactClassLoader.getClassLoader().equals(classLoader))
        .findFirst()
        .map(ArtifactClassLoader::getArtifactId);
  }

  private void checkClassLoaderId(String classLoaderId) {
    checkArgument(!isEmpty(requireNonNull(classLoaderId)), "artifactId cannot be empty");
  }
}
