/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static org.mule.runtime.api.artifact.ArtifactType.DOMAIN;
import static org.mule.runtime.api.config.FeatureFlaggingService.FEATURE_FLAGGING_SERVICE_KEY;
import static org.mule.runtime.ast.api.util.MuleAstUtils.emptyArtifact;
import static org.mule.runtime.config.api.ArtifactContextFactory.createArtifactContextFactory;
import static org.mule.runtime.core.api.config.ConfigurationBuilder.getMinimalConfigurationBuilder;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.ACTION_ON_MULE_ARTIFACT_DEPLOYMENT_NULL;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.CLASS_LOADER_REPOSITORY_CANNOT_BE_NULL;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.CLASS_LOADER_REPOSITORY_WAS_NOT_SET;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.EXECUTION_CLASSLOADER_WAS_NOT_SET;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.INSTALLATION_DIRECTORY_MUST_BE_A_DIRECTORY;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.MULE_CONTEXT_ARTIFACT_PROPERTIES_CANNOT_BE_NULL;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.ONLY_APPLICATIONS_OR_POLICIES_ARE_ALLOWED_TO_HAVE_A_PARENT_ARTIFACT;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.SERVICE_CONFIGURATOR_CANNOT_BE_NULL;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.SERVICE_REPOSITORY_CANNOT_BE_NULL;
import static org.mule.runtime.module.deployment.impl.internal.artifact.ArtifactContextBuilder.newBuilder;
import static org.mule.runtime.module.artifact.internal.util.test.DeploymentTestingFeatures.ALWAYS_ON_FEATURE;
import static org.mule.test.allure.AllureConstants.DeploymentConfiguration.FeatureFlaggingStory.FEATURE_FLAGGING;

import static java.lang.Thread.currentThread;
import static java.util.Optional.empty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.config.api.ArtifactContextFactory;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.ConfigurationException;
import org.mule.runtime.core.api.config.DefaultMuleConfiguration;
import org.mule.runtime.core.api.context.DefaultMuleContextFactory;
import org.mule.runtime.core.internal.registry.DefaultRegistry;
import org.mule.runtime.deployment.model.api.DeployableArtifact;
import org.mule.runtime.deployment.model.api.artifact.ArtifactConfigurationProcessor;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContextConfiguration;
import org.mule.runtime.module.artifact.api.classloader.ClassLoaderRepository;
import org.mule.tck.config.TestServicesConfigurationBuilder;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junitpioneer.jupiter.ClearSystemProperty;
import org.mockito.invocation.InvocationOnMock;

import io.qameta.allure.Issue;
import io.qameta.allure.Story;

class ArtifactContextBuilderTestCase extends AbstractMuleTestCase {

  @TempDir
  public static File temporaryFolder;

  @BeforeAll
  public static void beforeClass() {
    // Ensure that the testing feature flags are registered.
    ALWAYS_ON_FEATURE.getClass();
  }

  @Test
  void emptyBuilder() throws Exception {
    ArtifactConfigurationProcessor artifactConfigurationProcessor = mock(ArtifactConfigurationProcessor.class);
    when(artifactConfigurationProcessor.createArtifactContext(any()))
        .thenAnswer(this::doCreateArtifactContext);

    MuleContext muleContext =
        newBuilder(new TestServicesConfigurationBuilder())
            .setExecutionClassloader(currentThread().getContextClassLoader())
            .setClassLoaderRepository(mock(ClassLoaderRepository.class))
            .setArtifactConfigurationProcessor(artifactConfigurationProcessor)
            .build().getMuleContext();
    assertThat(muleContext, notNullValue());
    assertThat(muleContext.isInitialised(), is(true));

    try {
      muleContext.start();
      assertThat(muleContext.isStarted(), is(true));
    } finally {
      muleContext.stop();
      muleContext.dispose();
    }
  }

  @Test
  void buildWithoutClassloader() throws Exception {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.build());
    assertThat(thrown.getMessage(), is(EXECUTION_CLASSLOADER_WAS_NOT_SET));
  }

  @Test
  void setNullArtifactProperties() {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.setArtifactProperties(null));
    assertThat(thrown.getMessage(), is(MULE_CONTEXT_ARTIFACT_PROPERTIES_CANNOT_BE_NULL));
  }

  @Test
  void setNullClassLoaderRepository() {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.setClassLoaderRepository(null));
    assertThat(thrown.getMessage(), is(CLASS_LOADER_REPOSITORY_CANNOT_BE_NULL));
  }

  @Test
  void setNullServiceConfigurator() {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.withServiceConfigurator(null));
    assertThat(thrown.getMessage(), is(SERVICE_CONFIGURATOR_CANNOT_BE_NULL));
  }

  @Test
  void buildWithoutClassloaderRepository() throws Exception {
    var builder = newBuilder().setExecutionClassloader(currentThread().getContextClassLoader());
    var thrown = assertThrows(NullPointerException.class, () -> builder.build());
    assertThat(thrown.getMessage(), is(CLASS_LOADER_REPOSITORY_WAS_NOT_SET));
  }

  @Test
  void buildWithActionOnMuleArtifactDeployment() throws Exception {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.setActionOnMuleArtifactDeployment(null));
    assertThat(thrown.getMessage(), is(ACTION_ON_MULE_ARTIFACT_DEPLOYMENT_NULL));
  }

  @Test
  void setRegularFileInstallationLocation() throws Exception {
    var builder = newBuilder();
    var nonDirFile = new File(temporaryFolder, "hi.txt");
    var thrown = assertThrows(IllegalArgumentException.class,
                              () -> builder.setArtifactInstallationDirectory(nonDirFile));
    assertThat(thrown.getMessage(), is(INSTALLATION_DIRECTORY_MUST_BE_A_DIRECTORY));
  }

  @Test
  void buildUsingDomainAndParentArtifact() throws Exception {
    var builder = newBuilder().setArtifactType(DOMAIN)
        .setExecutionClassloader(currentThread().getContextClassLoader()).setParentArtifact(mock(DeployableArtifact.class))
        .setClassLoaderRepository(mock(ClassLoaderRepository.class));
    var thrown = assertThrows(IllegalStateException.class, () -> builder.build());
    assertThat(thrown.getMessage(), is(ONLY_APPLICATIONS_OR_POLICIES_ARE_ALLOWED_TO_HAVE_A_PARENT_ARTIFACT));
  }

  @Test
  void setNullServiceRepository() {
    var builder = newBuilder();
    var thrown = assertThrows(NullPointerException.class, () -> builder.setServiceRepository(null));
    assertThat(thrown.getMessage(), is(SERVICE_REPOSITORY_CANNOT_BE_NULL));
  }

  @Test
  @Story(FEATURE_FLAGGING)
  @Issue("MULE-19402")
  void buildSettingLegacyFeatureFlag() throws Exception {
    ArtifactConfigurationProcessor artifactConfigurationProcessor = mock(ArtifactConfigurationProcessor.class);
    when(artifactConfigurationProcessor.createArtifactContext(any()))
        .thenAnswer(inv -> {
          ArtifactContextConfiguration configBuilder = inv.getArgument(0, ArtifactContextConfiguration.class);

          ArtifactContext artifactContext = mock(ArtifactContext.class);
          when(artifactContext.getRegistry())
              .thenReturn(new DefaultRegistry(configBuilder.getMuleContext()));
          return artifactContext;
        });

    ArtifactContext artifactContext = newBuilder(new TestServicesConfigurationBuilder())
        .setExecutionClassloader(currentThread().getContextClassLoader())
        .setClassLoaderRepository(mock(ClassLoaderRepository.class))
        .setArtifactConfigurationProcessor(artifactConfigurationProcessor)
        .build();
    FeatureFlaggingService featureFlaggingService = (FeatureFlaggingService) artifactContext.getRegistry()
        .lookupByName(FEATURE_FLAGGING_SERVICE_KEY).get();
    assertThat(featureFlaggingService.isEnabled(ALWAYS_ON_FEATURE), is(true));
  }

  @Test
  @ClearSystemProperty(key = MULE_HOME_DIRECTORY_PROPERTY)
  void muleConfigurationStandaloneDirectory() throws Exception {
    ArtifactConfigurationProcessor artifactConfigurationProcessor = mock(ArtifactConfigurationProcessor.class);
    when(artifactConfigurationProcessor.createArtifactContext(any()))
        .thenAnswer(this::doCreateArtifactContext);

    System.setProperty(MULE_HOME_DIRECTORY_PROPERTY, "thehome");

    final var testServicesConfigurationBuilder = new TestServicesConfigurationBuilder();
    ArtifactContext artifactContext = newBuilder(testServicesConfigurationBuilder)
        .setExecutionClassloader(currentThread().getContextClassLoader())
        .setClassLoaderRepository(mock(ClassLoaderRepository.class))
        .setArtifactConfigurationProcessor(artifactConfigurationProcessor)
        .setDataFolderName("sarasa")
        .build();

    final var muleContext = artifactContext.getMuleContext();
    DefaultMuleConfiguration mutableConfig = ((DefaultMuleConfiguration) muleContext.getConfiguration());
    assertThat(mutableConfig.getWorkingDirectory(), endsWith("thehome/./.mule/sarasa"));
  }

  private ArtifactContext doCreateArtifactContext(InvocationOnMock inv) {
    try {
      ArtifactContextConfiguration artifactContextConfiguration = inv.getArgument(0, ArtifactContextConfiguration.class);

      ArtifactContextFactory configurationBuilder = createArtifactContextFactory(emptyArtifact(),
                                                                                 artifactContextConfiguration
                                                                                     .getArtifactProperties(),
                                                                                 artifactContextConfiguration
                                                                                     .getArtifactType(),
                                                                                 artifactContextConfiguration
                                                                                     .isEnableLazyInitialization(),
                                                                                 artifactContextConfiguration
                                                                                     .isAddToolingObjectsToRegistry(),
                                                                                 artifactContextConfiguration
                                                                                     .getServiceConfigurators(),
                                                                                 empty());

      configurationBuilder.configure(artifactContextConfiguration.getMuleContext());

      return configurationBuilder.createArtifactContext();
    } catch (ConfigurationException e) {
      throw new MuleRuntimeException(e);
    }
  }

}
