/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.plugin;

import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;
import static org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModelBuilder.forMavenProject;
import static org.mule.test.allure.AllureConstants.ArtifactPatchingFeature.ARTIFACT_PATCHING;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.PATCHED_ARTIFACT_DESCRIPTORS;

import static java.lang.System.setProperty;
import static java.util.Arrays.stream;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.not;

import org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModel;
import org.mule.runtime.module.artifact.activation.api.descriptor.DeployableArtifactDescriptorFactory;
import org.mule.runtime.module.artifact.activation.api.descriptor.DomainDescriptorResolver;
import org.mule.runtime.module.artifact.activation.internal.descriptor.DefaultDeployableArtifactDescriptorFactory;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.junitpioneer.jupiter.ClearSystemProperty;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(ARTIFACT_PATCHING)
@Story(PATCHED_ARTIFACT_DESCRIPTORS)
class DeployableDescriptorsWithPatchedPluginsTestCase extends AbstractMuleTestCase {

  private static final String PATCHES_LOCATION = "/lib/patches/mule-artifact-patches";

  @ParameterizedTest
  @ValueSource(strings = {"apps/with-patched-artifacts", "domains/with-patched-artifacts"})
  @ClearSystemProperty(key = MULE_HOME_DIRECTORY_PROPERTY)
  void createDeployableDescriptorWithPatchedPlugins(String deployableProjectFolder) throws URISyntaxException, IOException {
    setProperty(MULE_HOME_DIRECTORY_PROPERTY, getDeployableFolder(deployableProjectFolder).getCanonicalPath());
    DeployableArtifactDescriptor deployableArtifactDescriptor = getDeployableArtifactDescriptor(deployableProjectFolder);

    ArtifactPluginDescriptor httpPlugin = deployableArtifactDescriptor.getPlugins()
        .stream()
        .filter(p -> p.getName().equals("HTTP"))
        .findFirst()
        .get();

    assertThat(stream(httpPlugin.getClassLoaderConfiguration().getUrls()).collect(toList()),
               hasItem(getClass().getClassLoader().getResource(deployableProjectFolder + PATCHES_LOCATION + "/http-patch.jar")));

    ArtifactPluginDescriptor dbPlugin = deployableArtifactDescriptor.getPlugins()
        .stream()
        .filter(p -> p.getName().equals("Database"))
        .findFirst()
        .get();

    assertThat(stream(dbPlugin.getClassLoaderConfiguration().getUrls()).map(URL::toString).collect(toList()),
               not(hasItem(getClass().getClassLoader()
                   .getResource(deployableProjectFolder + PATCHES_LOCATION + "/db-patch.jar"))));
  }

  private DeployableArtifactDescriptor getDeployableArtifactDescriptor(String deployableProjectFolder) throws URISyntaxException {
    boolean isApplication = deployableProjectFolder.startsWith("apps");
    if (isApplication) {
      return createApplicationDescriptor(deployableProjectFolder);
    } else {
      return createDomainDescriptor(deployableProjectFolder);
    }
  }

  protected static final DeployableArtifactDescriptorFactory deployableArtifactDescriptorFactory =
      new DefaultDeployableArtifactDescriptorFactory();

  protected DomainDescriptor createDomainDescriptor(String domainPath) throws URISyntaxException {
    DeployableProjectModel model = getDeployableProjectModel(domainPath);

    return deployableArtifactDescriptorFactory.createDomainDescriptor(model, emptyMap());
  }

  protected ApplicationDescriptor createApplicationDescriptor(String appPath) throws URISyntaxException {
    return createApplicationDescriptor(appPath, null);
  }

  protected ApplicationDescriptor createApplicationDescriptor(String appPath, DomainDescriptorResolver domainDescriptorResolver)
      throws URISyntaxException {
    DeployableProjectModel model = getDeployableProjectModel(appPath);

    return deployableArtifactDescriptorFactory.createApplicationDescriptor(model, emptyMap(),
                                                                           domainDescriptorResolver);
  }

  protected DeployableProjectModel getDeployableProjectModel(String deployablePath) throws URISyntaxException {
    return forMavenProject(getDeployableFolder(deployablePath), false, false)
        .build();
  }

  protected File getDeployableFolder(String appPath) throws URISyntaxException {
    return new File(getClass().getClassLoader().getResource(appPath).toURI());
  }


}
