/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.plugin;

import static org.mule.runtime.api.artifact.ArtifactType.PLUGIN;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor.EXTENSION_BUNDLE_TYPE;
import static org.mule.runtime.module.deployment.impl.internal.plugin.BundlePluginDependenciesResolver.MULE_HTTP_CONNECTOR_ARTIFACT_ID;
import static org.mule.runtime.module.deployment.impl.internal.plugin.BundlePluginDependenciesResolver.MULE_HTTP_CONNECTOR_GROUP_ID;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.CLASSLOADING_ISOLATION;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.ARTIFACT_DESCRIPTORS;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.CLASSLOADER_CONFIGURATION;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.beans.HasPropertyWithValue.hasProperty;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.StringContains.containsString;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.hamcrest.MockitoHamcrest.argThat;

import org.mule.runtime.deployment.model.api.plugin.resolver.DuplicateExportedPackageException;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginResolutionError;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorFactory;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDependency;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleScope;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderConfiguration;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderConfiguration.ClassLoaderConfigurationBuilder;
import org.mule.runtime.module.deployment.impl.internal.artifact.DefaultArtifactDescriptorFactoryProvider;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import io.qameta.allure.Feature;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;

@Feature(CLASSLOADING_ISOLATION)
@Stories({@Story(ARTIFACT_DESCRIPTORS), @Story(CLASSLOADER_CONFIGURATION)})
class BundlePluginDependenciesResolverTestCase extends AbstractMuleTestCase {

  private static final String FOO_PLUGIN = "foo";
  private static final String BAZ_PLUGIN = "baz";
  private static final String BAR_PLUGIN = "bar";
  private static final String ECHO_PLUGIN = "echo";
  public static final String DEPENDENCY_PROVIDER_ERROR_MESSAGE = "Bundle URL should have been resolved for %s.";

  @TempDir
  public static File temporaryFolder;

  private static BundleDescriptor FOO_BUNDLE_DESCRIPTOR;
  private static BundleDescriptor BAZ_BUNDLE_DESCRIPTOR;
  private static BundleDescriptor LATEST_BAZ_BUNDLE_DESCRIPTOR;
  private static BundleDescriptor BAR_BUNDLE_DESCRIPTOR;
  private static BundleDescriptor ECHO_BUNDLE_DESCRIPTOR;
  private static BundleDescriptor LATEST_ECHO_BUNDLE_DESCRIPTOR;

  private static BundleDependency FOO_PLUGIN_DESCRIPTOR;
  private static BundleDependency BAZ_PLUGIN_DESCRIPTOR;
  private static BundleDependency LATEST_BAZ_PLUGIN_DESCRIPTOR;
  private static BundleDependency BAR_PLUGIN_DESCRIPTOR;
  private static BundleDependency ECHO_PLUGIN_DESCRIPTOR;

  @BeforeAll
  public static void before() {
    FOO_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(FOO_PLUGIN, "1.0");
    BAZ_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(BAZ_PLUGIN, "1.0");
    LATEST_BAZ_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(BAZ_PLUGIN, "1.1");
    BAR_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(BAR_PLUGIN, "1.0");
    ECHO_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(ECHO_PLUGIN, "1.0");
    LATEST_ECHO_BUNDLE_DESCRIPTOR = createTestBundleDescriptor(ECHO_PLUGIN, "1.1");

    FOO_PLUGIN_DESCRIPTOR = createBundleDependency(FOO_BUNDLE_DESCRIPTOR);
    BAZ_PLUGIN_DESCRIPTOR = createBundleDependency(BAZ_BUNDLE_DESCRIPTOR, true);
    LATEST_BAZ_PLUGIN_DESCRIPTOR = createBundleDependency(LATEST_BAZ_BUNDLE_DESCRIPTOR, true);
    BAR_PLUGIN_DESCRIPTOR = createBundleDependency(BAR_BUNDLE_DESCRIPTOR);
    ECHO_PLUGIN_DESCRIPTOR = createBundleDependency(ECHO_BUNDLE_DESCRIPTOR);
  }

  private static BundleDependency createBundleDependency(BundleDescriptor bundleDescriptor) {
    return createBundleDependency(bundleDescriptor, false);
  }

  private static BundleDependency createBundleDependency(BundleDescriptor bundleDescriptor, boolean createBundleUri) {
    final BundleDependency.Builder builder = new BundleDependency.Builder();
    builder.setDescriptor(bundleDescriptor);
    builder.setScope(BundleScope.COMPILE);
    if (createBundleUri) {
      builder.setBundleUri(new File(temporaryFolder, bundleDescriptor.getArtifactFileName()).toURI());
    }
    return builder.build();
  }

  private static BundleDescriptor createTestBundleDescriptor(String artifactId, String version) {
    return new BundleDescriptor.Builder().setGroupId("test").setArtifactId(artifactId).setVersion(version)
        .setType(EXTENSION_BUNDLE_TYPE).setClassifier(PLUGIN.getMavenArtifactClassifier()).build();
  }

  private final ArtifactPluginDescriptor fooPlugin = newArtifactPluginDescriptor(FOO_PLUGIN);
  private final ArtifactPluginDescriptor barPlugin = newArtifactPluginDescriptor(BAR_PLUGIN);
  private final ArtifactPluginDescriptor bazPlugin = newArtifactPluginDescriptor(BAZ_PLUGIN);
  private final ArtifactPluginDescriptor latestBazPlugin = newArtifactPluginDescriptor(BAZ_PLUGIN);
  private final ArtifactPluginDescriptor echoPlugin = newArtifactPluginDescriptor(ECHO_PLUGIN);
  private final ArtifactPluginDescriptor latestEchoPlugin = newArtifactPluginDescriptor(ECHO_PLUGIN);

  private PluginDependenciesResolver dependenciesResolver;

  private ArtifactDescriptorFactory artifactDescriptorFactory;

  @BeforeEach
  public void setUp() throws Exception {
    fooPlugin.setBundleDescriptor(FOO_BUNDLE_DESCRIPTOR);
    bazPlugin.setBundleDescriptor(BAZ_BUNDLE_DESCRIPTOR);
    latestBazPlugin.setBundleDescriptor(LATEST_BAZ_BUNDLE_DESCRIPTOR);
    barPlugin.setBundleDescriptor(BAR_BUNDLE_DESCRIPTOR);
    echoPlugin.setBundleDescriptor(ECHO_BUNDLE_DESCRIPTOR);
    latestEchoPlugin.setBundleDescriptor(LATEST_ECHO_BUNDLE_DESCRIPTOR);

    artifactDescriptorFactory = mock(ArtifactDescriptorFactory.class);
    dependenciesResolver =
        new DefaultArtifactDescriptorFactoryProvider().createBundlePluginDependenciesResolver(artifactDescriptorFactory);
  }

  private static ArtifactPluginDescriptor newArtifactPluginDescriptor(String name) {
    return new ArtifactPluginDescriptor(name);
  }

  @Test
  void resolvesIndependentPlugins() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, barPlugin, fooPlugin);
  }

  @Test
  void resolvesPluginOrderedDependency() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, barPlugin);
  }

  @Test
  void resolvesPluginDisorderedDependency() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {barPlugin, fooPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, barPlugin);
  }

  @Test
  void resolvesPluginDependencyWithCompatibleMinorVersion() throws Exception {

    ArtifactPluginDescriptor updatedFooPlugin = new ArtifactPluginDescriptor(FOO_PLUGIN);
    updatedFooPlugin.setBundleDescriptor(createTestBundleDescriptor(FOO_PLUGIN, "1.1"));
    ArtifactPluginDescriptor[] descriptors = {updatedFooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(createBundleDependency(FOO_BUNDLE_DESCRIPTOR))).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, updatedFooPlugin, barPlugin);
  }

  @Test
  void resolvesPluginDependencyWithSnapshotMinorVersion() throws Exception {

    ArtifactPluginDescriptor updatedFooPlugin = new ArtifactPluginDescriptor(FOO_PLUGIN);
    updatedFooPlugin.setBundleDescriptor(createTestBundleDescriptor(FOO_PLUGIN, "1.1-SNAPSHOT"));
    ArtifactPluginDescriptor[] descriptors = {updatedFooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(createBundleDependency(FOO_BUNDLE_DESCRIPTOR))).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, updatedFooPlugin, barPlugin);
  }

  @Test
  void resolvesSnapshotPluginDependencyWithCompatibleMinorVersion() throws Exception {

    ArtifactPluginDescriptor updatedFooPlugin = new ArtifactPluginDescriptor(FOO_PLUGIN);
    updatedFooPlugin.setBundleDescriptor(createTestBundleDescriptor(FOO_PLUGIN, "1.1"));
    ArtifactPluginDescriptor[] descriptors = {updatedFooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    BundleDescriptor fooBundleDescriptor = createTestBundleDescriptor(FOO_PLUGIN, "1.0-SNAPSHOT");
    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(createBundleDependency(fooBundleDescriptor))).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, updatedFooPlugin, barPlugin);
  }

  @Test
  void doesNotResolvesPluginDependencyWithIncompatibleMajorVersion() throws Exception {

    ArtifactPluginDescriptor majorUpdatedFooPlugin = new ArtifactPluginDescriptor(FOO_PLUGIN);
    majorUpdatedFooPlugin.setBundleDescriptor(createTestBundleDescriptor(FOO_PLUGIN, "2.0"));
    ArtifactPluginDescriptor[] descriptors = {majorUpdatedFooPlugin, barPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(createBundleDependency(FOO_BUNDLE_DESCRIPTOR))).build());

    assertThrows(PluginResolutionError.class, () -> dependenciesResolver.resolve(emptySet(), pluginDescriptors, true));
  }


  @Test
  void doesNotResolvesPluginDependencyWithIncompatibleMinorVersion() throws Exception {
    ArtifactPluginDescriptor majorUpdatedFooPlugin = new ArtifactPluginDescriptor(FOO_PLUGIN);
    majorUpdatedFooPlugin.setBundleDescriptor(createTestBundleDescriptor(FOO_PLUGIN, "1.1"));
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);

    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(createBundleDependency(majorUpdatedFooPlugin.getBundleDescriptor()))).build());

    assertThrows(PluginResolutionError.class, () -> dependenciesResolver.resolve(emptySet(), pluginDescriptors, true));
  }

  @Test
  void detectsUnresolvablePluginDependency() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {fooPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    fooPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(BAR_PLUGIN_DESCRIPTOR)).build());

    var thrown =
        assertThrows(PluginResolutionError.class, () -> dependenciesResolver.resolve(emptySet(), pluginDescriptors, true));
    assertThat(thrown.getMessage(), is(format(DEPENDENCY_PROVIDER_ERROR_MESSAGE, BAR_BUNDLE_DESCRIPTOR)));
  }

  @Test
  public void resolvesTransitiveDependencies() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin, bazPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(BAZ_PLUGIN_DESCRIPTOR)).build());
    bazPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, bazPlugin, barPlugin);
  }

  @Test
  void resolvesMultipleDependencies() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {bazPlugin, barPlugin, fooPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(BAZ_PLUGIN_DESCRIPTOR)).build());
    bazPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, bazPlugin, barPlugin);
  }

  @Test
  void resolvesPluginWithNewestVersionOnDependency() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {barPlugin, bazPlugin, fooPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(LATEST_BAZ_PLUGIN_DESCRIPTOR)).build());
    bazPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());
    latestBazPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    when(artifactDescriptorFactory
        .create(argThat(hasProperty("absolutePath", endsWith(latestBazPlugin.getBundleDescriptor().getArtifactFileName()))),
                any(Optional.class))).thenReturn(latestBazPlugin);

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, bazPlugin, barPlugin);
  }

  @Test
  void resolvesDependenciesTwoVersionWhenLatestComesFromTransitiveMinor() throws Exception {
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, latestEchoPlugin, bazPlugin};
    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);
    latestEchoPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(BAZ_PLUGIN_DESCRIPTOR)).build());
    fooPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(ECHO_PLUGIN_DESCRIPTOR))
        .build());
    echoPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder()
        .dependingOn(singleton(LATEST_BAZ_PLUGIN_DESCRIPTOR)).build());

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, bazPlugin, latestEchoPlugin, fooPlugin);
  }

  @Test
  void sanitizesDependantPluginExportedPackages() throws Exception {
    fooPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().exportingPackages(getFooExportedPackages()).build());

    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR))
        .exportingPackages(getBarExportedPackages()).build());
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, barPlugin);
    assertPluginExportedPackages(fooPlugin, "org.foo", "org.foo.mule");
    assertPluginExportedPackages(barPlugin, "org.bar", "org.baz", "org.bar.mule");
  }

  @Test
  void sanitizesTransitiveDependantPluginExportedPackages() throws Exception {
    fooPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().exportingPackages(getFooExportedPackages()).build());

    bazPlugin.setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().exportingPackages(getBazExportedPackages())
        .dependingOn(singleton(FOO_PLUGIN_DESCRIPTOR)).build());

    barPlugin.setClassLoaderConfiguration(new ClassLoaderConfiguration.ClassLoaderConfigurationBuilder()
        .exportingPackages(getBarExportedPackages())
        .dependingOn(singleton(BAZ_PLUGIN_DESCRIPTOR)).build());
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin, bazPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);

    final List<ArtifactPluginDescriptor> resolvedPluginDescriptors = dependenciesResolver.resolve(
                                                                                                  emptySet(),
                                                                                                  pluginDescriptors, true);

    assertResolvedPlugins(resolvedPluginDescriptors, fooPlugin, bazPlugin, barPlugin);
    assertPluginExportedPackages(fooPlugin, "org.foo", "org.foo.mule");
    assertPluginExportedPackages(bazPlugin, "org.baz");
    assertPluginExportedPackages(barPlugin, "org.bar", "org.bar.mule");
  }

  @Test
  void detectsDuplicateExportedPackagesOnIndependentPlugins() throws Exception {
    fooPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().exportingPackages(getFooExportedPackages()).build());

    barPlugin
        .setClassLoaderConfiguration(new ClassLoaderConfigurationBuilder().exportingPackages(getBarExportedPackages()).build());
    ArtifactPluginDescriptor[] descriptors = {fooPlugin, barPlugin};

    final List<ArtifactPluginDescriptor> pluginDescriptors = asList(descriptors);

    Map<String, List<String>> pluginsPerPackage = new HashMap<>();
    pluginsPerPackage.put("org.foo", asList("bar, foo"));
    pluginsPerPackage.put("org.foo.mule", asList("bar, foo"));
    String expectedErrorMessage = new DuplicateExportedPackageException(pluginsPerPackage).getMessage();

    var thrown = assertThrows(DuplicateExportedPackageException.class,
                              () -> dependenciesResolver.resolve(emptySet(), pluginDescriptors, true));
    assertThat(thrown.getMessage(), containsString(expectedErrorMessage));
  }

  @Test
  void providedPluginsHaveOldestVersionOfSamePluginInPolicy() {
    dependenciesResolver.resolve(singleton(bazPlugin), singletonList(latestBazPlugin), false);
  }

  @Test
  void providedPluginsHaveOldestVersionOfSamePluginInDomain() {
    assertThrows(IllegalStateException.class,
                 () -> dependenciesResolver.resolve(singleton(bazPlugin), singletonList(latestBazPlugin), true));
  }

  @Test
  void providedPluginsHaveOldestVersionOfSameHttpPluginInPolicy() {
    BundleDescriptor httpDescriptor1_0 = new BundleDescriptor.Builder()
        .setGroupId(MULE_HTTP_CONNECTOR_GROUP_ID)
        .setArtifactId(MULE_HTTP_CONNECTOR_ARTIFACT_ID)
        .setVersion("1.0.0")
        .build();
    BundleDescriptor httpDescriptor1_1 = new BundleDescriptor.Builder()
        .setGroupId(MULE_HTTP_CONNECTOR_GROUP_ID)
        .setArtifactId(MULE_HTTP_CONNECTOR_ARTIFACT_ID)
        .setVersion("1.1.0")
        .build();
    ArtifactPluginDescriptor httpPluginDescriptor1_0 = newArtifactPluginDescriptor("HTTP");
    httpPluginDescriptor1_0.setBundleDescriptor(httpDescriptor1_0);
    ArtifactPluginDescriptor httpPluginDescriptor1_1 = newArtifactPluginDescriptor("HTTP");
    httpPluginDescriptor1_1.setBundleDescriptor(httpDescriptor1_1);

    var thrown = assertThrows(IllegalStateException.class, () -> dependenciesResolver
        .resolve(singleton(httpPluginDescriptor1_0), singletonList(httpPluginDescriptor1_1), false));
    assertThat(thrown.getMessage(),
               is("Incompatible version of plugin 'HTTP' (org.mule.connectors:mule-http-connector) found. Artifact requires version '1.1.0' but context provides version '1.0.0'"));
  }

  private Set<String> getBarExportedPackages() {
    final Set<String> barExportedClassPackages = new HashSet<>();
    barExportedClassPackages.add("org.bar");
    barExportedClassPackages.add("org.baz");
    barExportedClassPackages.add("org.foo");
    barExportedClassPackages.add("org.foo.mule");
    barExportedClassPackages.add("org.bar.mule");
    return barExportedClassPackages;
  }

  private Set<String> getFooExportedPackages() {
    final Set<String> fooExportedClassPackages = new HashSet<>();
    fooExportedClassPackages.add("org.foo");
    fooExportedClassPackages.add("org.foo.mule");
    return fooExportedClassPackages;
  }

  private Set<String> getBazExportedPackages() {
    final Set<String> bazExportedClassPackages = new HashSet<>();
    bazExportedClassPackages.add("org.baz");
    return bazExportedClassPackages;
  }

  private void assertResolvedPlugins(List<ArtifactPluginDescriptor> resolvedPluginDescriptors,
                                     ArtifactPluginDescriptor... expectedPluginDescriptors) {
    assertThat(resolvedPluginDescriptors.size(), equalTo(expectedPluginDescriptors.length));

    assertThat(resolvedPluginDescriptors, contains(expectedPluginDescriptors));
  }

  private void assertPluginExportedPackages(ArtifactPluginDescriptor pluginDescriptor, String... exportedPackages) {
    assertThat(pluginDescriptor.getClassLoaderConfiguration().getExportedPackages().size(), equalTo(exportedPackages.length));
    assertThat(pluginDescriptor.getClassLoaderConfiguration().getExportedPackages(), containsInAnyOrder(exportedPackages));
  }
}
