/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static java.util.Optional.empty;

import org.mule.runtime.api.artifact.ArtifactType;
import org.mule.runtime.module.artifact.activation.api.deployable.DeployableProjectModel;
import org.mule.runtime.module.artifact.api.Artifact;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;
import java.util.Properties;

/**
 * Generic Factory for an {@link Artifact}.
 */
public interface ArtifactFactory<D extends DeployableArtifactDescriptor, T extends Artifact<D>> {

  /**
   * @return the directory of the Artifact. Usually this directory contains the Artifact resources
   * @deprecated Use {@link #getArtifactPath()} instead.
   */
  @Deprecated(forRemoval = false, since = "4.11")
  File getArtifactDir();

  /**
   * @return the directory of the Artifact. Usually this directory contains the Artifact resources
   */
  Path getArtifactPath();

  /**
   * Creates an Artifact
   *
   * @param artifactDir          directory where the artifact is located
   * @param deploymentProperties deployment properties for configuration management
   * @return the newly created Artifact
   */
  default T createArtifact(File artifactLocation, Optional<Properties> deploymentProperties) throws IOException {
    return createArtifact(artifactLocation.toPath(), empty(), deploymentProperties);
  }

  /**
   * Creates an Artifact
   *
   * @param artifactDir          directory where the artifact is located
   * @param model                the model to use instead of loading it from the artifact, if present
   * @param deploymentProperties deployment properties for configuration management
   * @return the newly created Artifact
   */
  T createArtifact(Path artifactLocation,
                   Optional<DeployableProjectModel> model,
                   Optional<Properties> deploymentProperties)
      throws IOException;

  /**
   * @return the kind of artifact this factory creates.
   */
  public ArtifactType getTargetArtifactType();

}
