/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.config.custom.ServiceConfigurator;
import org.mule.runtime.container.internal.splash.ArtifactShutdownSplashScreen;
import org.mule.runtime.container.internal.splash.ServerShutdownSplashScreen;
import org.mule.runtime.container.internal.splash.SplashScreen;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.ConfigurationBuilder;
import org.mule.runtime.core.api.config.ConfigurationException;
import org.mule.runtime.core.api.context.notification.MuleContextListener;
import org.mule.runtime.core.internal.context.DefaultMuleContext;

import org.slf4j.Logger;
import org.slf4j.event.Level;

final class SplashScreenConfigurationBuilder implements ConfigurationBuilder {

  private static final Logger SPLASH_LOGGER = getLogger("org.mule.runtime.core.internal.logging");

  private final Level splashScrenLevel;

  public SplashScreenConfigurationBuilder(Level splashScrenLevel) {
    this.splashScrenLevel = splashScrenLevel;
  }

  @Override
  public void configure(MuleContext muleContext) throws ConfigurationException {
    if (muleContext instanceof DefaultMuleContext dmc) {
      dmc.addListener(new MuleContextListener() {

        @Override
        public void onStart(MuleContext context, Registry registry) {
          if (SPLASH_LOGGER.isEnabledForLevel(splashScrenLevel)) {
            SplashScreen startupScreen = buildStartupSplash(context);
            SPLASH_LOGGER.atLevel(splashScrenLevel).log(startupScreen.toString());
          }
        }

        @Override
        public void onStop(MuleContext context, Registry registry) {
          if ((context.getStartDate() > 0) && SPLASH_LOGGER.isEnabledForLevel(splashScrenLevel)) {
            SplashScreen shutdownScreen = buildShutdownSplash(context);
            SPLASH_LOGGER.atLevel(splashScrenLevel).log(shutdownScreen.toString());
          }
        }

        @Override
        public void onInitialization(MuleContext context, Registry registry) {
          // Nothing to do
        }

        @Override
        public void onCreation(MuleContext context) {
          // Nothing to do
        }
      });
    }
  }

  SplashScreen buildStartupSplash(MuleContext muleContext) {
    SplashScreen startupScreen = createSplashScreen(muleContext);
    startupScreen.setHeader(muleContext);
    startupScreen.setFooter(muleContext);
    return startupScreen;
  }

  SplashScreen buildShutdownSplash(MuleContext muleContext) {
    SplashScreen shutdownScreen = createSplashScreen(muleContext);
    shutdownScreen.setHeader(muleContext);
    return shutdownScreen;
  }

  private SplashScreen createSplashScreen(MuleContext muleContext) {
    return muleContext.getConfiguration().isContainerMode() ? new ArtifactShutdownSplashScreen()
        : new ServerShutdownSplashScreen();
  }

  @Override
  public void addServiceConfigurator(ServiceConfigurator serviceConfigurator) {
    // Nothing to do
  }
}
