/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.artifact;

import static org.mule.runtime.api.artifact.ArtifactType.APP;
import static org.mule.runtime.module.deployment.impl.internal.artifact.MavenClassLoaderConfigurationLoaderConfigurationTestCase.MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION;
import static org.mule.tck.MuleTestUtils.testWithSystemProperties;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.CLASSLOADING_ISOLATION;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.CLASSLOADER_CONFIGURATION;
import static org.mule.test.allure.AllureConstants.ClassloadingIsolationFeature.ClassloadingIsolationStory.CLASSLOADER_CONFIGURATION_LOADER;

import static java.util.Collections.emptyMap;
import static java.util.Map.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.IntStream.range;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.hamcrest.core.IsIterableContaining.hasItem;
import static org.junit.jupiter.api.Assertions.assertThrows;

import io.qameta.allure.Feature;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junitpioneer.jupiter.ClearSystemProperty;
import org.mule.runtime.globalconfig.api.GlobalConfigLoader;

import java.io.File;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Feature(CLASSLOADING_ISOLATION)
@Stories({@Story(CLASSLOADER_CONFIGURATION_LOADER), @Story(CLASSLOADER_CONFIGURATION)})
@ClearSystemProperty(key = MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION)
class MavenClassLoaderConfigurationLoaderConfigurationTestCase extends MavenClassLoaderConfigurationLoaderTestCase {

  public static final String MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION = "muleRuntimeConfig.maven.repositoryLocation";

  @TempDir
  public Path tempDir;

  private File artifactFile;

  private String repositoryLocation;

  @BeforeEach
  public void before() throws URISyntaxException {
    var repositoryLocation = tempDir.toAbsolutePath().toString();
    System.setProperty(MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION,
                       repositoryLocation);

    artifactFile = getApplicationFolder("apps/single-dependency");
  }

  @Test
  void noMavenConfiguration() throws Exception {
    testWithSystemProperties(getMuleFreeSystemProperties(), () -> {
      GlobalConfigLoader.reset(); // Change local repository path

      var thrown =
          assertThrows(RuntimeException.class, () -> mavenClassLoaderConfigurationLoader.load(artifactFile, emptyMap(), APP));
      assertThat(thrown.getCause(), instanceOf(DependencyResolutionException.class));
    });
  }

  @Test
  void changeMavenConfiguration() throws Exception {
    Map<String, String> properties = getMuleFreeSystemProperties();
    properties.put(MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION, repositoryLocation);
    testWithSystemProperties(properties, () -> {
      GlobalConfigLoader.reset(); // Change local repository path
      assertThrows(Exception.class, () -> mavenClassLoaderConfigurationLoader.load(artifactFile, emptyMap(), APP));
    });
    properties.put("muleRuntimeConfig.maven.repositories.mavenCentral.url", "https://repo.maven.apache.org/maven2/");

    Map<String, Object> attributes =
        of(org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.class.getName(),
           new org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder()
               .setGroupId("groupId")
               .setArtifactId("artifactId")
               .setVersion("1.0.0")
               .setType("jar")
               .setClassifier("mule-application")
               .build());

    testWithSystemProperties(properties, () -> range(1, 10).parallel()
        .forEach(number -> {
          GlobalConfigLoader.reset();
          try {
            var dependencies = mavenClassLoaderConfigurationLoader
                .load(artifactFile, attributes, APP)
                .getDependencies();
            assertThat(dependencies,
                       hasItem(hasProperty("descriptor", (hasProperty("artifactId", equalTo("commons-collections"))))));
          } catch (Exception e) {
            throw new RuntimeException(e);
          }
        }));
  }

  private Map<String, String> getMuleFreeSystemProperties() {
    // Find any System property for muleRuntimeConfig from previous executions...
    final List<String> muleRuntimeConfig =
        System.getProperties().stringPropertyNames().stream()
            .filter(propertyName -> propertyName.startsWith("muleRuntimeConfig")
                && !propertyName.equals(MULE_RUNTIME_CONFIG_MAVEN_REPOSITORY_LOCATION))
            .collect(toList());

    Map<String, String> properties = new HashMap<>();
    muleRuntimeConfig.forEach(property -> properties.put(property, null));
    return properties;
  }

}
