/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.plugin;

import static java.util.Objects.requireNonNull;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * Allows to extend the attributes defined for a {@link org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor} when it
 * is being loaded by {@link org.mule.runtime.module.deployment.impl.internal.maven.MavenBundleDescriptorLoader} for plugins in
 * order to use the effective {@link BundleDescriptor} resolved from the artifact that declares the plugin. This allows to work
 * with SNAPSHOT versions in order to propagate the timestamped version of the SNAPSHOT artifact.
 *
 * @since 4.2.0
 */
public class PluginExtendedBundleDescriptorAttributes implements Map<String, Object> {

  private final Map<String, Object> delegate;
  private final BundleDescriptor pluginBundleDescriptor;

  public PluginExtendedBundleDescriptorAttributes(Map<String, Object> attributes, BundleDescriptor pluginBundleDescriptor) {
    this.delegate = attributes;
    requireNonNull(pluginBundleDescriptor, "pluginBundleDescriptor cannot be null");
    this.pluginBundleDescriptor = pluginBundleDescriptor;
  }

  public BundleDescriptor getPluginBundleDescriptor() {
    return pluginBundleDescriptor;
  }

  @Override
  public int size() {
    return delegate.size();
  }

  @Override
  public boolean isEmpty() {
    return delegate.isEmpty();
  }

  @Override
  public boolean containsKey(Object key) {
    return delegate.containsKey(key);
  }

  @Override
  public boolean containsValue(Object value) {
    return delegate.containsValue(value);
  }

  @Override
  public Object get(Object key) {
    return delegate.get(key);
  }

  @Override
  public String put(String key, Object value) {
    return (String) delegate.put(key, value);
  }

  @Override
  public Object remove(Object key) {
    return delegate.remove(key);
  }

  @Override
  public void putAll(Map<? extends String, ?> m) {
    delegate.putAll(m);
  }

  @Override
  public void clear() {
    delegate.clear();
  }

  @Override
  public Set<String> keySet() {
    return delegate.keySet();
  }

  @Override
  public Collection<Object> values() {
    return delegate.values();
  }

  @Override
  public Set<Entry<String, Object>> entrySet() {
    return delegate.entrySet();
  }

}
