/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.impl.internal.plugin;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderConfiguration;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderConfigurationLoader;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * Allows to extend the attributes defined for a {@link ClassLoaderConfiguration} when it is being loaded by
 * {@link ClassLoaderConfigurationLoader} for plugins in order to define in which deployable artifact the plugin is declared.
 *
 * @since 4.2.0
 */
public class PluginExtendedClassLoaderConfigurationAttributes implements Map<String, Object> {

  private final Map<String, Object> delegate;
  private final ArtifactDescriptor deployableArtifactDescriptor;

  /**
   * Creates an instance of this extended attributes for the given descriptor.
   *
   * @param originalAttributes           the original {@link Map} of attributes. No null.
   * @param deployableArtifactDescriptor {@link ArtifactDescriptor} which declares the plugin dependency. Not null.
   */
  public PluginExtendedClassLoaderConfigurationAttributes(Map<String, Object> originalAttributes,
                                                          ArtifactDescriptor deployableArtifactDescriptor) {
    this.delegate = originalAttributes;
    requireNonNull(deployableArtifactDescriptor, "deployableArtifactDescriptor cannot be null");
    this.deployableArtifactDescriptor = deployableArtifactDescriptor;
  }

  /**
   * @return the {@link ArtifactDescriptor} which declares the dependency to the plugin.
   */
  public ArtifactDescriptor getDeployableArtifactDescriptor() {
    return deployableArtifactDescriptor;
  }

  @Override
  public int size() {
    return delegate.size();
  }

  @Override
  public boolean isEmpty() {
    return delegate.isEmpty();
  }

  @Override
  public boolean containsKey(Object key) {
    return delegate.containsKey(key);
  }

  @Override
  public boolean containsValue(Object value) {
    return delegate.containsValue(value);
  }

  @Override
  public Object get(Object key) {
    return delegate.get(key);
  }

  @Override
  public Object put(String key, Object value) {
    return delegate.put(key, value);
  }

  @Override
  public Object remove(Object key) {
    return delegate.remove(key);
  }

  @Override
  public void putAll(Map<? extends String, ? extends Object> m) {
    delegate.putAll(m);
  }

  @Override
  public void clear() {
    delegate.clear();
  }

  @Override
  public Set<String> keySet() {
    return delegate.keySet();
  }

  @Override
  public Collection<Object> values() {
    return delegate.values();
  }

  @Override
  public Set<Entry<String, Object>> entrySet() {
    return delegate.entrySet();
  }

}
