/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.internal;

import static org.mule.runtime.api.util.Preconditions.checkArgument;

import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.api.StartupListener;
import org.mule.runtime.module.deployment.internal.util.DebuggableReentrantLock;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public abstract class AbstractDeploymentService implements DeploymentService {

  private static final Logger LOGGER = LoggerFactory.getLogger(AbstractDeploymentService.class);

  private final ReentrantLock deploymentLock = new DebuggableReentrantLock(true);

  /**
   * TODO: move to setter as in previous version.
   */
  private final CompositeDeploymentListener applicationDeploymentListener = new CompositeDeploymentListener();
  private final CompositeDeploymentListener domainDeploymentListener = new CompositeDeploymentListener();
  private final CompositeDeploymentListener domainBundleDeploymentListener = new CompositeDeploymentListener();

  private final List<StartupListener> startupListeners = new CopyOnWriteArrayList<>();

  @Override
  public final void addDeploymentListener(DeploymentListener listener) {
    checkArgument(listener != null, "Listener cannot be null");
    applicationDeploymentListener.addDeploymentListener(listener);
  }

  @Override
  public final void removeDeploymentListener(DeploymentListener listener) {
    checkArgument(listener != null, "Listener cannot be null");
    applicationDeploymentListener.removeDeploymentListener(listener);
  }

  @Override
  public final void addDomainBundleDeploymentListener(DeploymentListener listener) {
    domainBundleDeploymentListener.addDeploymentListener(listener);
  }

  @Override
  public final void removeDomainBundleDeploymentListener(DeploymentListener listener) {
    domainBundleDeploymentListener.removeDeploymentListener(listener);
  }

  @Override
  public final void addDomainDeploymentListener(DeploymentListener listener) {
    domainDeploymentListener.addDeploymentListener(listener);
  }

  @Override
  public final void removeDomainDeploymentListener(DeploymentListener listener) {
    domainDeploymentListener.removeDeploymentListener(listener);
  }

  @Override
  public final void addStartupListener(StartupListener listener) {
    startupListeners.add(listener);
  }

  @Override
  public final void removeStartupListener(StartupListener listener) {
    startupListeners.remove(listener);
  }

  public final void notifyStartupListeners() {
    for (StartupListener listener : startupListeners) {
      try {
        listener.onAfterStartup();
      } catch (Throwable t) {
        LOGGER.error("Error executing startup listener {}", listener, t);
      }
    }
  }

  protected Map<String, String> additionalDeploymentProperties() {
    return emptyMap();
  }

  protected final Optional<Properties> createAdditionalDeploymentProperties() {
    var additionalDeploymentProperties = additionalDeploymentProperties();
    if (additionalDeploymentProperties == null || additionalDeploymentProperties.isEmpty()) {
      return empty();
    }

    Properties properties = new Properties();
    properties.putAll(additionalDeploymentProperties);
    return of(properties);
  }

  @Override
  public final ReentrantLock getLock() {
    return deploymentLock;
  }

  public DeploymentListener getApplicationDeploymentListener() {
    return applicationDeploymentListener;
  }

  public DeploymentListener getDomainDeploymentListener() {
    return domainDeploymentListener;
  }

  public DeploymentListener getDomainBundleDeploymentListener() {
    return domainBundleDeploymentListener;
  }

  public List<StartupListener> getStartupListeners() {
    return startupListeners;
  }

}
