/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.test.internal;

import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.bridgeMethodExtensionPlugin;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.helloExtensionV1Plugin;
import static org.mule.test.allure.AllureConstants.LeakPrevention.LEAK_PREVENTION;
import static org.mule.test.allure.AllureConstants.LeakPrevention.LeakPreventionMetaspace.METASPACE_LEAK_PREVENTION_ON_REDEPLOY;

import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;

import org.mule.runtime.module.deployment.impl.internal.builder.ArtifactPluginFileBuilder;

import java.util.HashSet;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Issues;
import io.qameta.allure.Story;

/**
 * Contains tests for leak prevention on the deployment process.
 */
@Feature(LEAK_PREVENTION)
@Story(METASPACE_LEAK_PREVENTION_ON_REDEPLOY)
@Issues({@Issue("W-13160893"), @Issue("MULE-17311"), @Issue("W-16226692")})
public class ClassLoaderLeakOnDeploymentTestCase extends ClassLoaderLeakTestCase {

  public static final Supplier<Set<ArtifactPluginFileBuilder>> BRIDGE_METHOD_PLUGIN =
      () -> new HashSet<>(singletonList(bridgeMethodExtensionPlugin));
  public static final Supplier<Set<ArtifactPluginFileBuilder>> HELLO_V1_PLUGIN =
      () -> new HashSet<>(singletonList(helloExtensionV1Plugin));
  public static final Supplier<Set<ArtifactPluginFileBuilder>> NO_PLUGINS = () -> emptySet();

  static Stream<Arguments> parameters() {
    return Stream.of(
                     Arguments.of("empty-config-1.0.0-mule-application",
                                  "empty-config", NO_PLUGINS),
                     Arguments.of("appWithExtensionPlugin-1.0.0-mule-application",
                                  "app-with-extension-plugin-config", HELLO_V1_PLUGIN),
                     Arguments.of("appWithExtensionPlugin-1.0.0-mule-application",
                                  "app-with-bridge-extension-plugin-config",
                                  BRIDGE_METHOD_PLUGIN));
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void undeploysApplicationDoesNotLeakClassloader(String appName, String xmlFile,
                                                            Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.undeploysApplicationDoesNotLeakClassloader(appName, xmlFile, applicationPlugins);
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void undeploysApplicationWithPoliciesDoesNotLeakClassloader(String appName, String xmlFile,
                                                                        Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.undeploysApplicationWithPoliciesDoesNotLeakClassloader(appName, xmlFile, applicationPlugins);
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void undeploysApplicationWithTracingDoesNotLeakClassloader(String appName, String xmlFile,
                                                                       Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.undeploysApplicationWithTracingDoesNotLeakClassloader(appName, xmlFile, applicationPlugins);
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void redeployByConfigChangePreviousAppEagerlyGCd(String appName, String xmlFile,
                                                             Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.redeployByConfigChangePreviousAppEagerlyGCd(appName, xmlFile, applicationPlugins);
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void redeployPreviousAppEagerlyGCd(String appName, String xmlFile,
                                               Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.redeployByConfigChangePreviousAppEagerlyGCd(appName, xmlFile, applicationPlugins);
  }

  @ParameterizedTest(name = "{displayName} AppName: {0}")
  @MethodSource("parameters")
  @Override
  protected void redeployByNamePreviousAppEagerlyGCd(String appName, String xmlFile,
                                                     Supplier<Set<ArtifactPluginFileBuilder>> applicationPlugins)
      throws Exception {
    super.redeployByConfigChangePreviousAppEagerlyGCd(appName, xmlFile, applicationPlugins);
  }
}
