/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.test.internal;

import static org.mule.runtime.module.deployment.internal.DeploymentDirectoryWatcher.CHANGE_CHECK_INTERVAL_PROPERTY;
import static org.mule.runtime.module.deployment.internal.MuleDeploymentService.PARALLEL_DEPLOYMENT_PROPERTY;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.callbackExtensionPlugin;
import static org.mule.test.allure.AllureConstants.ArtifactDeploymentFeature.DOMAIN_DEPLOYMENT;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.core.IsNull.nullValue;

import static org.mockito.Mockito.reset;

import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.runtime.module.deployment.impl.internal.builder.DomainBundleFileBuilder;
import org.mule.runtime.module.deployment.impl.internal.builder.DomainFileBuilder;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.junitpioneer.jupiter.ClearSystemProperty;
import org.junitpioneer.jupiter.SetSystemProperty;

import io.qameta.allure.Feature;

/**
 * Contains test for domain bundle deployment
 */
@Feature(DOMAIN_DEPLOYMENT)
@SetSystemProperty(key = CHANGE_CHECK_INTERVAL_PROPERTY, value = "100000")
@ClearSystemProperty(key = PARALLEL_DEPLOYMENT_PROPERTY)
class DomainBundleDeploymentTestCase extends AbstractDeploymentTestCase {

  @Test
  void deploysDomainBundle() throws Exception {
    ApplicationFileBuilder applicationFileBuilder = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder)
        .dependingOn(callbackExtensionPlugin).dependingOn(dummyDomainFileBuilder);
    DomainBundleFileBuilder domainBundleFileBuilder =
        new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDeploymentSuccess(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertApplicationDeploymentSuccess(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());

    assertDomainDir(NONE, new String[] {dummyDomainFileBuilder.getId()}, true);
    assertAppsDir(NONE, new String[] {dummyAppDescriptorFileBuilder.getId()}, true);
  }

  @Test
  void failsToDeployDomainBundleWithCorruptedDomain() throws Exception {
    ApplicationFileBuilder applicationFileBuilder = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder)
        .dependingOn(dummyDomainFileBuilder);
    DomainBundleFileBuilder domainBundleFileBuilder =
        new DomainBundleFileBuilder(new DomainFileBuilder("dummy-domain")).containing(applicationFileBuilder);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentFailure(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDeploymentFailure(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertDeploymentFailure(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());
  }

  @Test
  void deploysDomainBundleWithCorruptedApp() throws Exception {
    ApplicationFileBuilder applicationFileBuilder = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder).corrupted();
    DomainBundleFileBuilder domainBundleFileBuilder =
        new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentFailure(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDomainDir(NONE, new String[] {dummyDomainFileBuilder.getId()}, true);

    assertDeploymentSuccess(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertDeploymentFailure(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());
  }

  @ParameterizedTest(name = "{displayName} - Parallel: {0}")
  @ValueSource(booleans = {false, true})
  void redeploysDomainBundle(boolean parallelDeployment) throws Exception {
    if (parallelDeployment) {
      System.setProperty(PARALLEL_DEPLOYMENT_PROPERTY, "");
    }

    ApplicationFileBuilder applicationFileBuilder = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder)
        .dependingOn(dummyDomainFileBuilder).dependingOn(callbackExtensionPlugin);
    DomainBundleFileBuilder domainBundleFileBuilder =
        new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDeploymentSuccess(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertApplicationDeploymentSuccess(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());

    reset(domainDeploymentListener);
    reset(domainBundleDeploymentListener);
    reset(applicationDeploymentListener);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDomainRedeploymentSuccess(dummyDomainFileBuilder.getId());
    assertApplicationRedeploymentSuccess(dummyAppDescriptorFileBuilder.getId());
  }

  @ParameterizedTest(name = "{displayName} - Parallel: {0}")
  @ValueSource(booleans = {false, true})
  void redeploysDomainBundleCausesUndeployOfRemovedApps(boolean parallelDeployment) throws Exception {
    if (parallelDeployment) {
      System.setProperty(PARALLEL_DEPLOYMENT_PROPERTY, "");
    }

    ApplicationFileBuilder applicationFileBuilder1 = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder)
        .dependingOn(callbackExtensionPlugin).dependingOn(dummyDomainFileBuilder);
    ApplicationFileBuilder applicationFileBuilder2 = new ApplicationFileBuilder(emptyAppFileBuilder)
        .dependingOn(callbackExtensionPlugin).dependingOn(dummyDomainFileBuilder);

    DomainBundleFileBuilder domainBundleFileBuilder = new DomainBundleFileBuilder(dummyDomainFileBuilder)
        .containing(applicationFileBuilder1).containing(applicationFileBuilder2);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDeploymentSuccess(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertApplicationDeploymentSuccess(applicationDeploymentListener, applicationFileBuilder1.getId());
    assertApplicationDeploymentSuccess(applicationDeploymentListener, applicationFileBuilder2.getId());

    reset(domainDeploymentListener);
    reset(domainBundleDeploymentListener);
    reset(applicationDeploymentListener);

    domainBundleFileBuilder = new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder1);
    addDomainBundleFromBuilder(domainBundleFileBuilder);

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDomainRedeploymentSuccess(dummyDomainFileBuilder.getId());
    assertApplicationRedeploymentSuccess(applicationFileBuilder1.getId());
    assertApplicationMissingOnBundleRedeployment(applicationFileBuilder2.getId());
  }

  @ParameterizedTest(name = "{displayName} - Parallel: {0}")
  @ValueSource(booleans = {false, true})
  void redeploysDomainBundleWithBrokenDomain(boolean parallelDeployment) throws Exception {
    if (parallelDeployment) {
      System.setProperty(PARALLEL_DEPLOYMENT_PROPERTY, "");
    }

    ApplicationFileBuilder applicationFileBuilder = new ApplicationFileBuilder(dummyAppDescriptorFileBuilder)
        .dependingOn(callbackExtensionPlugin).dependingOn(dummyDomainFileBuilder);
    DomainBundleFileBuilder domainBundleFileBuilder =
        new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder);

    addDomainBundleFromBuilder(domainBundleFileBuilder);

    startDeployment();

    assertDeploymentSuccess(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDeploymentSuccess(domainDeploymentListener, dummyDomainFileBuilder.getId());
    assertApplicationDeploymentSuccess(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());

    reset(domainDeploymentListener);
    reset(domainBundleDeploymentListener);
    reset(applicationDeploymentListener);

    dummyDomainFileBuilder = new DomainFileBuilder(dummyDomainFileBuilder).corrupted();
    domainBundleFileBuilder = new DomainBundleFileBuilder(dummyDomainFileBuilder).containing(applicationFileBuilder);
    addDomainBundleFromBuilder(domainBundleFileBuilder);

    assertDeploymentFailure(domainBundleDeploymentListener, domainBundleFileBuilder.getId());
    assertDomainRedeploymentFailure(dummyDomainFileBuilder.getId());
    assertRedeploymentFailure(applicationDeploymentListener, dummyAppDescriptorFileBuilder.getId());
    assertThat(deploymentService.findApplication(dummyAppDescriptorFileBuilder.getId()), is(nullValue()));
  }

  private void addDomainBundleFromBuilder(DomainBundleFileBuilder domainBundleFileBuilder) throws Exception {
    addPackedDomainFromBuilder(domainBundleFileBuilder, null);
  }
}
