/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.test.internal;

import static org.mule.runtime.container.api.discoverer.ModuleDiscoverer.EXPORTED_CLASS_PACKAGES_PROPERTY;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.EXPORTED_PACKAGES;
import static org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptorConstants.EXPORTED_RESOURCES;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.barUtils1ClassFile;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.callbackExtensionLoadingResource;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.callbackExtensionPlugin;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.echoTestClassFile;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.loadsAppResourceCallbackClassFile;
import static org.mule.runtime.module.deployment.test.internal.artifacts.TestArtifactsCatalog.pluginEcho2ClassFile;
import static org.mule.runtime.module.deployment.test.internal.util.Utils.getResourceFile;
import static org.mule.test.allure.AllureConstants.ArtifactDeploymentFeature.APP_DEPLOYMENT;

import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.runtime.module.deployment.impl.internal.builder.ArtifactPluginFileBuilder;
import org.mule.tck.util.CompilerUtils.SingleClassCompiler;

import java.nio.file.Path;

import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;

@Feature(APP_DEPLOYMENT)
public class ApplicationDeploymentLocalPackagesResourcesTestCase extends AbstractApplicationDeploymentTestCase {

  private String classloaderModelVersion;

  @ParameterizedTest(name = "{displayName} - classloaderModelVersion: {0}")
  @ValueSource(strings = {"1.0.0", "1.1.0", "1.2.0"})
  @Issue("MULE-13756")
  @Description("Tests that code called form plugin's Processor cannot access internal resources/packages of the application")
  void deploysAppWithLocalPackageAndPlugin(String classloaderModelVersion) throws Exception {
    this.classloaderModelVersion = classloaderModelVersion;

    ArtifactPluginFileBuilder loadsAppResourcePlugin = new ArtifactPluginFileBuilder("loadsAppResourcePlugin")
        .configuredWith(EXPORTED_CLASS_PACKAGES_PROPERTY, "org.foo")
        .containingClass(loadsAppResourceCallbackClassFile, "org/foo/LoadsAppResourceCallback.class");

    ApplicationFileBuilder nonExposingAppFileBuilder = appFileBuilder("non-exposing-app")
        .configuredWith(EXPORTED_PACKAGES, "org.bar")
        .configuredWith(EXPORTED_RESOURCES, "test-resource.txt")
        .definedBy("app-with-loads-app-resource-plugin-config.xml")
        .dependingOn(callbackExtensionPlugin)
        .containingClass(barUtils1ClassFile, "org/bar/BarUtils.class")
        .containingClass(pluginEcho2ClassFile, "org/foo/echo/Plugin2Echo.class")
        .containingResource("test-resource.txt", "test-resource.txt")
        .containingResource("test-resource.txt", "test-resource-not-exported.txt")
        .dependingOn(loadsAppResourcePlugin);

    addPackedAppFromBuilder(nonExposingAppFileBuilder);

    startDeployment();

    assertApplicationDeploymentSuccess(applicationDeploymentListener, nonExposingAppFileBuilder.getId());

    executeApplicationFlow("main");
  }

  @ParameterizedTest(name = "{displayName} - classloaderModelVersion: {0}")
  @ValueSource(strings = {"1.0.0", "1.1.0", "1.2.0"})
  @Issue("MULE-13756")
  @Description("Tests that code called form application's Processor can access internal resources/packages of the application")
  void deploysAppWithLocalPackage(String classloaderModelVersion) throws Exception {
    this.classloaderModelVersion = classloaderModelVersion;

    ApplicationFileBuilder nonExposingAppFileBuilder = appFileBuilder("non-exposing-app")
        .configuredWith(EXPORTED_PACKAGES, "org.bar")
        .configuredWith(EXPORTED_RESOURCES, "test-resource.txt")
        .definedBy("app-with-loads-app-resource-plugin-config.xml")
        .dependingOn(callbackExtensionLoadingResource)
        .containingClass(barUtils1ClassFile, "org/bar/BarUtils.class")
        .containingClass(pluginEcho2ClassFile, "org/foo/echo/Plugin2Echo.class")
        .containingResource("test-resource.txt", "test-resource.txt")
        .containingResource("test-resource.txt", "test-resource-not-exported.txt");

    addPackedAppFromBuilder(nonExposingAppFileBuilder);

    startDeployment();

    assertApplicationDeploymentSuccess(applicationDeploymentListener, nonExposingAppFileBuilder.getId());

    executeApplicationFlow("main");
  }

  @ParameterizedTest(name = "{displayName} - classloaderModelVersion: {0}")
  @ValueSource(strings = {"1.0.0", "1.1.0", "1.2.0"})
  @Issue("MULE-13756")
  @Description("Tests that code called form plugin's ProcessorInterceptor cannot access internal resources/packages of the application")
  void deploysAppWithLocalPackageAndPluginWithInterceptors(String classloaderModelVersion) throws Exception {
    this.classloaderModelVersion = classloaderModelVersion;

    Path loadsAppResourceInterceptorFactoryClassFile = new SingleClassCompiler()
        .useClassPath()
        .compile(getResourceFile("/org/foo/LoadsAppResourceInterceptorFactory.java"));
    Path loadsAppResourceInterceptorClassFile = new SingleClassCompiler()
        .useClassPath()
        .compile(getResourceFile("/org/foo/LoadsAppResourceInterceptor.java"));

    ArtifactPluginFileBuilder loadsAppResourceInterceptorPlugin =
        new ArtifactPluginFileBuilder("loadsAppResourceInterceptorPlugin")
            .configuredWith(EXPORTED_CLASS_PACKAGES_PROPERTY, "org.lalala")
            .containingClass(loadsAppResourceInterceptorFactoryClassFile, "org/foo/LoadsAppResourceInterceptorFactory.class")
            .containingClass(loadsAppResourceInterceptorClassFile, "org/foo/LoadsAppResourceInterceptor.class")
            .containingResource("registry-bootstrap-loads-app-resource-pif.properties",
                                "META-INF/org/mule/runtime/core/config/registry-bootstrap.properties");

    ApplicationFileBuilder nonExposingAppFileBuilder = appFileBuilder("non-exposing-app")
        .configuredWith(EXPORTED_PACKAGES, "org.bar")
        .configuredWith(EXPORTED_RESOURCES, "test-resource.txt")
        .definedBy("app-with-plugin-bootstrap.xml")
        .containingClass(barUtils1ClassFile, "org/bar/BarUtils.class")
        .containingClass(echoTestClassFile, "org/foo/EchoTest.class")
        .containingResource("test-resource.txt", "test-resource.txt")
        .containingResource("test-resource.txt", "test-resource-not-exported.txt")
        .dependingOn(loadsAppResourceInterceptorPlugin);

    addPackedAppFromBuilder(nonExposingAppFileBuilder);

    startDeployment();

    assertApplicationDeploymentSuccess(applicationDeploymentListener, nonExposingAppFileBuilder.getId());

    executeApplicationFlow("main");
  }

  @ParameterizedTest(name = "{displayName} - classloaderModelVersion: {0}")
  @ValueSource(strings = {"1.0.0", "1.1.0", "1.2.0"})
  @Issue("MULE-13756")
  @Description("Tests that code called form application's ProcessorInterceptor can access internal resources/packages of the application")
  void deploysAppWithInterceptorsAndLocalPackage(String classloaderModelVersion) throws Exception {
    this.classloaderModelVersion = classloaderModelVersion;

    Path loadsOwnResourceInterceptorFactoryClassFile = new SingleClassCompiler()
        .useClassPath()
        .compile(getResourceFile("/org/foo/LoadsOwnResourceInterceptorFactory.java"));
    Path loadsOwnResourceInterceptorClassFile = new SingleClassCompiler()
        .useClassPath()
        .compile(getResourceFile("/org/foo/LoadsOwnResourceInterceptor.java"));

    ApplicationFileBuilder nonExposingAppFileBuilder = appFileBuilder("non-exposing-app")
        .configuredWith(EXPORTED_PACKAGES, "org.bar")
        .configuredWith(EXPORTED_RESOURCES, "test-resource.txt")
        .definedBy("app-with-interceptor.xml")
        .containingClass(loadsOwnResourceInterceptorFactoryClassFile, "org/foo/LoadsOwnResourceInterceptorFactory.class")
        .containingClass(loadsOwnResourceInterceptorClassFile, "org/foo/LoadsOwnResourceInterceptor.class")
        .containingClass(barUtils1ClassFile, "org/bar/BarUtils.class")
        .containingClass(echoTestClassFile, "org/foo/EchoTest.class")
        .containingResource("test-resource.txt", "test-resource.txt")
        .containingResource("test-resource.txt", "test-resource-not-exported.txt");

    addPackedAppFromBuilder(nonExposingAppFileBuilder);

    startDeployment();

    assertApplicationDeploymentSuccess(applicationDeploymentListener, nonExposingAppFileBuilder.getId());

    executeApplicationFlow("main");
  }

  @Override
  protected ApplicationFileBuilder appFileBuilder(final String artifactId) {
    return new ApplicationFileBuilder(artifactId)
        .withClassloaderModelVersion(classloaderModelVersion);
  }

}
