/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.test.internal;

import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.mock;

import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;
import org.mule.runtime.module.deployment.internal.ArchiveDeployer;
import org.mule.runtime.module.deployment.internal.DeploymentDirectoryWatcher;
import org.mule.runtime.module.deployment.internal.DomainBundleArchiveDeployer;
import org.mule.runtime.module.deployment.internal.util.ObservableList;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

@SmallTest
class DeploymentDirectoryWatcherTestCase extends AbstractMuleTestCase {

  private DeploymentDirectoryWatcher watcher;

  @BeforeEach
  void setUp() {
    DomainBundleArchiveDeployer domainBundleDeployer = mock(DomainBundleArchiveDeployer.class);
    ArchiveDeployer<DomainDescriptor, Domain> domainArchiveDeployer = mock(ArchiveDeployer.class);
    ArchiveDeployer<ApplicationDescriptor, Application> applicationArchiveDeployer = mock(ArchiveDeployer.class);
    ObservableList<Domain> domains = new ObservableList<>();
    ObservableList<Application> applications = new ObservableList<>();
    Supplier<SchedulerService> schedulerServiceSupplier = () -> mock(SchedulerService.class);
    ReentrantLock deploymentLock = new ReentrantLock();

    watcher = new DeploymentDirectoryWatcher(empty(), domainBundleDeployer, domainArchiveDeployer,
                                             applicationArchiveDeployer, domains, applications,
                                             schedulerServiceSupplier, deploymentLock);
  }

  @ParameterizedTest(name = "[{index}] {0} -> {1}")
  @MethodSource("provideArtifactNameTestCases")
  void extractArtifactName(String fileName, String expectedArtifactName) {
    Path path = Paths.get("/apps/" + fileName);
    String result = watcher.extractArtifactName(path);
    assertThat(result, is(expectedArtifactName));
  }

  static Stream<Arguments> provideArtifactNameTestCases() {
    return Stream.of(
                     Arguments.of("my-app.jar", "my-app"),
                     Arguments.of("my-app-1.0.0.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-mule-application.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-SNAPSHOT.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-SNAPSHOT-mule-application.jar", "my-app"),
                     Arguments.of("my-app-2.5.10.jar", "my-app"),
                     Arguments.of("my-app-1.0.jar", "my-app"),
                     Arguments.of("my-app-1.jar", "my-app"),
                     Arguments.of("my-complex-app-name-1.0.0.jar", "my-complex-app-name"),
                     Arguments.of("my-complex-app-name-1.0.0-mule-application.jar", "my-complex-app-name"),
                     Arguments.of("my_app_name-1.0.0.jar", "my_app_name"),
                     Arguments.of("my-app-1.0.0-20231110.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-RC1.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-beta.jar", "my-app"),
                     Arguments.of("my-app-2.0.0-alpha-1.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-FINAL.jar", "my-app"),
                     Arguments.of("my-app-1.0.0-M1.jar", "my-app"),
                     Arguments.of("my-app123-1.0.0.jar", "my-app123"),
                     Arguments.of("MyApp-1.0.0.jar", "MyApp"),
                     Arguments.of("my-app-1.2.3.4.5.jar", "my-app"));
  }
}

