/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.internal;

import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.DEPLOYMENT_FAILED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.STARTED;
import static org.mule.runtime.module.deployment.internal.DefaultArchiveDeployer.START_ARTIFACT_ON_DEPLOYMENT_PROPERTY;

import static java.lang.String.valueOf;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.module.artifact.api.Artifact;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

/**
 * Utility to hook callbacks just before and after a domain zip is redeployed in Mule.
 */
public final class DomainDeploymentTemplate implements ArtifactDeploymentTemplate {

  private Collection<Application> domainApplications = emptyList();
  private Map<Application, ApplicationStatus> appStatusPreRedeployment;
  private final DefaultArchiveDeployer<ApplicationDescriptor, Application> applicationDeployer;
  private final DeploymentService deploymentservice;
  private final DeploymentListener applicationDeploymentListener;

  public DomainDeploymentTemplate(DefaultArchiveDeployer<ApplicationDescriptor, Application> applicationDeployer,
                                  DeploymentService deploymentservice,
                                  DeploymentListener applicationDeploymentListener) {
    this.applicationDeployer = applicationDeployer;
    this.deploymentservice = deploymentservice;
    this.applicationDeploymentListener = applicationDeploymentListener;
  }

  /**
   * Undeploys all applications that use this domain.
   */
  @Override
  public void preRedeploy(Artifact domain) {
    if (domain instanceof Domain) {
      appStatusPreRedeployment = new HashMap<>();
      domainApplications = deploymentservice.findDomainApplications(domain.getArtifactName());
      for (Application domainApplication : domainApplications) {
        appStatusPreRedeployment.put(domainApplication, domainApplication.getStatus());
        applicationDeploymentListener.onRedeploymentStart(domainApplication.getArtifactName());
        applicationDeployer.undeployArtifactWithoutUninstall(domainApplication);
      }
    }
  }

  /**
   * Deploys applications that were undeployed when {@link #preRedeploy(Artifact)} was called.
   */
  @Override
  public void postRedeploy(Artifact domain) {
    if (domain != null && !domainApplications.isEmpty()) {
      RuntimeException firstException = null;
      for (Application domainApplication : domainApplications) {
        applicationDeployer.preTrackArtifact(domainApplication);
        if (applicationDeployer.isUpdatedZombieArtifact(domainApplication.getArtifactName())) {
          try {
            applicationDeployer.deployExplodedArtifact(domainApplication.getDescriptor().getRootFolder().toPath(), empty(),
                                                       getArtifactStatusProperties(appStatusPreRedeployment
                                                           .get(domainApplication)));
            applicationDeploymentListener.onRedeploymentSuccess(domainApplication.getArtifactName());
          } catch (RuntimeException e) {
            applicationDeploymentListener.onRedeploymentFailure(domainApplication.getArtifactName(), e);
            if (firstException == null) {
              firstException = e;
            }
          }
        }
      }
      if (firstException != null) {
        throw firstException;
      }
    }
    domainApplications = Collections.emptyList();
  }

  private Optional<Properties> getArtifactStatusProperties(ApplicationStatus applicationStatus) {
    Properties properties = new Properties();
    boolean startArtifact = applicationStatus.equals(STARTED) || applicationStatus.equals(DEPLOYMENT_FAILED);
    properties.setProperty(START_ARTIFACT_ON_DEPLOYMENT_PROPERTY, valueOf(startArtifact));
    return of(properties);
  }
}
