/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.test.internal;

import static com.github.valfirst.slf4jtest.TestLoggerFactory.getTestLogger;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.internal.DeploymentStatusTracker;
import org.mule.runtime.module.deployment.internal.StartupSummaryDeploymentListener;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.github.valfirst.slf4jtest.TestLogger;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class StartupSummaryDeploymentListenerTestCase extends AbstractMuleTestCase {

  private final TestLogger loggerDefaultMuleApplication = getTestLogger("org.mule.runtime.core.internal.logging");

  private DeploymentService deploymentService;
  private DeploymentStatusTracker tracker;
  private StartupSummaryDeploymentListener startupSummaryDeploymentListener;

  @BeforeEach
  public void setUp() {
    loggerDefaultMuleApplication.clearAll();

    deploymentService = mock(DeploymentService.class);
    tracker = new DeploymentStatusTracker();
    startupSummaryDeploymentListener = new StartupSummaryDeploymentListener(tracker, deploymentService);
  }

  @Test
  void singleApp() {
    tracker.getApplicationDeploymentStatusTracker().onDeploymentSuccess("app1A");

    startupSummaryDeploymentListener.onAfterStartup();
    assertThat(loggerDefaultMuleApplication.getAllLoggingEvents().get(0).getMessage(),
               is("""

                   **********************************************************************
                   *            - - + APPLICATION + - -            * - - + STATUS + - - *
                   **********************************************************************
                   * app1A                                         * DEPLOYED           *
                   **********************************************************************
                                          """));
  }

  @Test
  void twoDomains() {
    var application = mock(Application.class);
    var domain = mock(Domain.class);
    when(domain.getArtifactName()).thenReturn("domain1");
    when(application.getDomain()).thenReturn(domain);
    when(deploymentService.findApplication("app1A")).thenReturn(application);

    tracker.getDomainDeploymentStatusTracker().onDeploymentSuccess("domain1");
    tracker.getDomainDeploymentStatusTracker().onDeploymentFailure("domain2", new Exception());

    tracker.getApplicationDeploymentStatusTracker().onDeploymentSuccess("app1A");

    startupSummaryDeploymentListener.onAfterStartup();
    assertThat(loggerDefaultMuleApplication.getAllLoggingEvents().get(0).getMessage(),
               is("""

                   **********************************************************************
                   *              - - + DOMAIN + - -               * - - + STATUS + - - *
                   **********************************************************************
                   * domain2                                       * FAILED             *
                   * domain1                                       * DEPLOYED           *
                   **********************************************************************

                   *******************************************************************************************************
                   *            - - + APPLICATION + - -            *       - - + DOMAIN + - -       * - - + STATUS + - - *
                   *******************************************************************************************************
                   * app1A                                         * domain1                        * DEPLOYED           *
                   *******************************************************************************************************
                                          """));
  }
}
