/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.deployment.internal;

import static java.lang.String.format;
import static java.nio.file.Files.list;
import static java.util.Optional.empty;

import org.mule.runtime.deployment.model.api.DeploymentException;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeploymentUtils {

  protected static final Logger LOGGER = LoggerFactory.getLogger(DeploymentUtils.class);

  public DeploymentUtils() {}

  public static List<Path> listFiles(Path directory, Predicate<Path> filter) {
    try (var content = list(directory)) {
      return content
          .filter(filter)
          .toList();
    } catch (IOException e) {
      throw new IllegalStateException(format("We got a null while listing the contents of director '%s'. Some common " +
          "causes for this is a lack of permissions to the directory or that it's being deleted concurrently",
                                             directory.getFileName()),
                                      e);
    }
  }

  public static void deployExplodedDomains(ArchiveDeployer<DomainDescriptor, Domain> domainDeployer) {
    List<Path> domains = listFiles(domainDeployer.getDeploymentDirectoryPath(), Files::isDirectory);
    deployExplodedDomains(domainDeployer, domains, empty());
  }

  public static void deployExplodedDomains(ArchiveDeployer<DomainDescriptor, Domain> domainDeployer, List<Path> domains,
                                           Optional<Properties> additionalDeploymentProperties) {
    domains.forEach(addedDomain -> {
      try {
        if (domainDeployer.isUpdatedZombieArtifact(addedDomain.getFileName().toString())) {
          domainDeployer.deployExplodedArtifact(addedDomain, additionalDeploymentProperties);
        }
      } catch (DeploymentException e) {
        LOGGER.error("Error deploying domain '{}'", addedDomain, e);
      }
    });
  }

}
