/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import org.mule.runtime.extension.api.error.ErrorTypeDefinition;

import java.util.Optional;

/**
 * Dynamic error type definition for custom error kinds.
 */
public class DynamicErrorTypeDefinition<T extends Enum<T>> implements ErrorTypeDefinition<T> {

  private final String errorTypeName;
  private final ErrorTypeDefinition<?> parent;

  /**
   * Creates a new dynamic error type definition.
   *
   * @param errorTypeName the name of the error type (part after "custom-" prefix)
   */
  public DynamicErrorTypeDefinition(String errorTypeName) {
    this(errorTypeName, null);
  }

  /**
   * Creates a new dynamic error type definition with a parent.
   *
   * @param errorTypeName the name of the error type (part after "custom-" prefix)
   * @param parent the parent error type definition (can be null for root errors)
   */
  public DynamicErrorTypeDefinition(String errorTypeName, ErrorTypeDefinition<?> parent) {
    this.errorTypeName = errorTypeName;
    this.parent = parent;
  }

  @Override
  public Optional<ErrorTypeDefinition<? extends Enum<?>>> getParent() {
    return Optional.ofNullable(parent);
  }

  @Override
  public String getType() {
    return errorTypeName;
  }

  /**
   * Gets the name of this error type.
   *
   * @return the error type name
   */
  public String getErrorTypeName() {
    return errorTypeName;
  }

  @Override
  public String toString() {
    return errorTypeName;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || getClass() != obj.getClass()) {
      return false;
    }
    DynamicErrorTypeDefinition<?> that = (DynamicErrorTypeDefinition<?>) obj;
    return errorTypeName.equals(that.errorTypeName);
  }

  @Override
  public int hashCode() {
    return errorTypeName.hashCode();
  }
}
