/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import static org.mule.runtime.extension.ic.internal.error.ConnectivityErrorModelParserFactory.createErrorModelParser;
import static org.mule.runtime.extension.ic.internal.error.ConnectivityErrorModelParserFactory.getDefaultErrorModelParser;
import static org.mule.runtime.extension.ic.internal.utils.StringUtils.isNullOrEmpty;

import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.extension.api.loader.parser.ErrorModelParser;
import org.mule.runtime.extension.ic.internal.parser.ConnectivityErrorModelParser;

import java.util.Optional;
import java.util.Set;

/**
 * Factory class for creating ErrorTypeDefinition instances from error models. This factory handles the conversion of connectivity
 * error models to Mule error type definitions.
 */
public class ErrorTypeDefinitionFactory {

  /**
   * Private constructor to prevent instantiation of utility class.
   */
  private ErrorTypeDefinitionFactory() {
    // Utility class - no instantiation needed
  }

  /**
   * Creates an ErrorTypeDefinition from an error model's kind.
   * <p>
   * Switch case similar to createErrorModelParserFromModel method within {@link ConnectivityErrorModelParserFactory} . Please add
   * case in mentioned class if changing it here.
   * </p>
   *
   * @param extensionName the extension Name.
   * @param kind the error kind.
   * @param errorModelSet the set of error models.
   * @return the created ErrorTypeDefinition.
   */
  public static ErrorTypeDefinition<?> createErrorTypeDefinition(String extensionName, String kind,
                                                                 Set<ErrorModel> errorModelSet) {

    if (isNullOrEmpty(kind)) {
      return MuleErrors.ANY;
    }

    // kind will always be of the form <statusCode> and <customVal>
    ErrorTypeDefinition<?> errorTypeDefinition = createErrorTypeDefinitionFromCustomLogic(extensionName, kind);

    // Matches any of the errorModel for this operations
    for (ErrorModel errorModel : errorModelSet) {
      if (errorModel.getType().equals(errorTypeDefinition.getType())) {
        // If any of the model matches with current errorType
        return errorTypeDefinition;
      }
    }

    // If none of them match with current errorType then derive default error based on statusCode or customVal
    // Error Hierarchy would be: if kind is statusCode then CLIENT_ERROR/SERVER_ERROR/ANY otherwise ANY
    return convertToErrorTypeDefinition(getDefaultErrorModelParser(kind, extensionName));
  }

  private static ErrorTypeDefinition<?> createErrorTypeDefinitionFromCustomLogic(String extensionName, String errorKind) {

    // same mapping needed as loader time
    ConnectivityErrorModelParser errorModelParser = createErrorModelParser(errorKind, extensionName);
    return convertToErrorTypeDefinition(errorModelParser);
  }

  private static ErrorTypeDefinition<?> convertToErrorTypeDefinition(ConnectivityErrorModelParser errorModelParser) {
    // Recursively handle parent hierarchy
    ErrorTypeDefinition<?> parentErrorType = null;
    Optional<ErrorModelParser> parentParserOptional = errorModelParser.getParent();
    if (parentParserOptional.isPresent()) {
      ConnectivityErrorModelParser parentParser = (ConnectivityErrorModelParser) parentParserOptional.get();
      parentErrorType = convertToErrorTypeDefinition(parentParser);
    }

    if (errorModelParser.isMuleError()) {
      return getMatchedMuleError(errorModelParser.getType(), parentErrorType);
    } else {
      return new DynamicErrorTypeDefinition<ConnectivityError>(errorModelParser.getType(), parentErrorType);
    }
  }

  private static ErrorTypeDefinition<?> getMatchedMuleError(String errorModelType, ErrorTypeDefinition<?> parentErrorType) {
    MuleErrors[] values = MuleErrors.values();
    for (MuleErrors muleError : values) {
      if (muleError.name().equals(errorModelType)) {
        return muleError;
      }
    }
    return new DynamicErrorTypeDefinition<MuleErrors>(errorModelType, parentErrorType);
  }
}
