/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.TypedException;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.core.api.config.i18n.CoreMessages;

public class ExceptionUtils {

  private ExceptionUtils() {}

  /**
   * Throws a TypedException for MULE:ANY errors.
   *
   * @param errorMessage the error message
   * @return a TypedException with MULE:ANY error type
   */
  public static TypedException throwTypedException(String errorMessage) {
    // Create MULE:ANY error type
    ErrorType muleAnyErrorType = createMuleAnyErrorType();

    // Create MuleException with proper message
    MuleException exception = createMuleException(errorMessage);

    // Create and throw TypedException
    return new TypedException(exception, muleAnyErrorType);
  }

  /**
   * Creates a MuleException with the given error message.
   *
   * @param errorMessage the error message
   * @return the MuleException
   */
  private static MuleException createMuleException(String errorMessage) {
    I18nMessage message = CoreMessages.createStaticMessage(errorMessage);
    return new MuleException(message) {

      @Override
      protected void setMessage(I18nMessage msg) {
        try {
          super.setMessage(msg);
        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      }
    };
  }

  /**
   * Creates a MULE:ANY error type.
   *
   * @return the MULE:ANY error type
   */
  private static ErrorType createMuleAnyErrorType() {
    return new ErrorType() {

      @Override
      public String getNamespace() {
        return ConnectivityErrorConstants.MULE_NAMESPACE;
      }

      @Override
      public ErrorType getParentErrorType() {
        return null;
      }

      @Override
      public String getIdentifier() {
        return ConnectivityErrorConstants.ANY_ERROR_TYPE;
      }

      @Override
      public String toString() {
        return ConnectivityErrorConstants.MULE_NAMESPACE + ":" + ConnectivityErrorConstants.ANY_ERROR_TYPE;
      }
    };
  }
}
