/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.loader;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;

import com.mulesoft.connectivity.mule.persistence.deserialize.loader.MuleModelDeserializer;
import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import com.google.gson.Gson;
import org.apache.commons.io.IOUtils;

/**
 * A utility class that helps to load the UC Connector.
 */
public class ConnectorLoaderHandler {

  private static final String URL_ERROR_MESSAGE =
      "Unable to load the connector metadata. There is no file present in the connector artifact in the path: '%s'";

  private ConnectorLoaderHandler() {}

  /**
   * Loads the MuleConnectorModel from the given modulePath.
   *
   * @param muleModelDeserializer the mule model serializer
   * @param modulePath the path of the resource to load.
   * @return the deserialized MuleConnectorModel.
   */
  public static MuleConnectorSerializableModel loadConnectorModel(MuleModelDeserializer muleModelDeserializer,
                                                                  String modulePath) {
    URL resource = getResourceAsURL(modulePath);
    if (resource == null) {
      throw new IllegalStateException(format(URL_ERROR_MESSAGE, modulePath));
    }

    String serializedModel = getResourceAsString(resource);
    return muleModelDeserializer.deserialize((Map<?, ?>) new Gson().fromJson(serializedModel, Object.class));
  }

  /**
   * Gets the resource content as a String.
   *
   * @param resource the URL of the resource.
   * @return the content of the resource as a String.
   */
  private static String getResourceAsString(URL resource) {
    try (BufferedInputStream resourceIS = new BufferedInputStream(resource.openStream())) {
      return IOUtils.toString(resourceIS, StandardCharsets.UTF_8);
    } catch (IOException e) {
      throw new RuntimeException("Unable to process the resource : " + e);
    }
  }

  /**
   * Gets the resource as a URL.
   *
   * @param resource the resource path.
   * @return the URL of the resource.
   */
  private static URL getResourceAsURL(String resource) {
    return currentThread().getContextClassLoader().getResource(resource);
  }
}
