/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.apache.commons.lang.StringUtils.EMPTY;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.parameter.ParameterRole;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.declaration.type.annotation.SemanticTermsTypeAnnotation;
import org.mule.runtime.extension.api.loader.parser.ValueProviderModelParser;
import org.mule.runtime.extension.ic.internal.parser.utils.AnnotationUtils;

import com.mulesoft.connectivity.mule.persistence.model.MuleValueProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderArgument;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderReferences;

import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Parameter model parser for interpreted connectivity extensions. Transforms object field types from LinkWeave models into Mule
 * extension parameter models.
 */
public class ConnectivityParameterModelParser implements ParameterModelParserWithDefaults {

  private final ObjectFieldType objectFieldType;
  private final Set<TypeAnnotation> annotations;
  private final MuleValueProviderSerializableModel muleValueProviderSerializableModel;
  private final SerializedProviderReferences providerReferences;


  // Parameters that should appear under the "Advanced" tab in the UI
  private static final Set<String> ADVANCED_TAB_PARAMETERS = Set.of("customHeaders", "customQueryParams");
  private static final Set<String> SENSITIVE_TERMS = Set.of("password", "secretToken", "apiKey");

  public ConnectivityParameterModelParser(ObjectFieldType param,
                                          MuleValueProviderSerializableModel valueProviderSerializableModel,
                                          SerializedProviderReferences providerRefs) {
    providerReferences = providerRefs;
    objectFieldType = param;
    annotations = objectFieldType.getValue().getAnnotations();
    this.muleValueProviderSerializableModel = valueProviderSerializableModel;
  }


  @Override
  public String getName() {
    return objectFieldType.getKey().getName().getLocalPart();
  }

  @Override
  public String getDescription() {
    return AnnotationUtils.findAnnotation(annotations, DescriptionAnnotation.class)
        .map(DescriptionAnnotation::getValue)
        .orElse(EMPTY);
  }

  @Override
  public MetadataType getType() {
    return objectFieldType.getValue();
  }

  @Override
  public boolean isRequired() {
    return objectFieldType.isRequired();
  }

  @Override
  public Object getDefaultValue() {
    return AnnotationUtils.findAnnotation(annotations, DefaultValueAnnotation.class)
        .map(DefaultValueAnnotation::getValue)
        .orElse(null);
  }

  @Override
  public ParameterRole getRole() {
    if (objectFieldType.getValue() instanceof ObjectType) {
      return ParameterRole.CONTENT;
    }
    return ParameterRole.BEHAVIOUR;
  }

  @Override
  public ExpressionSupport getExpressionSupport() {
    return ExpressionSupport.SUPPORTED;
  }

  @Override
  public Optional<LayoutModel> getLayoutModel() {
    if (ADVANCED_TAB_PARAMETERS.contains(getName())) {
      return Optional.of(LayoutModel.builder().tabName(Placement.ADVANCED_TAB).build());
    }

    return AnnotationUtils.findAnnotation(annotations, SemanticTermsTypeAnnotation.class)
        .flatMap(annotation -> annotation.getSemanticTerms().stream()
            .flatMap(term -> SENSITIVE_TERMS.stream().filter(term::contains))
            .findFirst()
            .map(__ -> LayoutModel.builder().asPassword().build()));
  }

  @Override
  public Optional<ValueProviderModelParser> getValueProviderModelParser() {
    if (muleValueProviderSerializableModel == null) {
      return Optional.empty();
    }
    // Extract provider arguments from the serialized provider references
    List<SerializedProviderArgument> serializedArguments =
        providerReferences.getValueProviderReference() != null ? providerReferences.getValueProviderReference().getArguments()
            : List.of();

    return Optional
        .of(new ConnectivityValueProviderParser(muleValueProviderSerializableModel,
                                                serializedArguments));
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    String displayName = AnnotationUtils.findAnnotation(annotations, LabelAnnotation.class)
        .map(LabelAnnotation::getValue)
        .orElse(objectFieldType.getKey().getName().getLocalPart());

    String description = AnnotationUtils.findAnnotation(annotations, DescriptionAnnotation.class)
        .map(DescriptionAnnotation::getValue)
        .orElse(EMPTY);

    return Optional.of(
                       DisplayModel.builder()
                           .displayName(displayName)
                           .summary(description)
                           .build());
  }

  @Override
  public Set<String> getSemanticTerms() {
    return AnnotationUtils.getSemanticTerms(annotations);
  }


}
