/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */

package org.mule.runtime.extension.ic.internal.runtime.operation.pagination;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.extension.ic.internal.runtime.connection.Connection.getErrorCauseIfUncheckedError;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getMediaType;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getRawValue;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.operation.ExecutionContext;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.extension.ic.internal.runtime.connection.Connection;
import org.mule.runtime.extension.ic.internal.utils.StreamingParameterUtils;

import com.mulesoft.connectivity.mule.api.Content;
import com.mulesoft.connectivity.mule.api.Page;
import com.mulesoft.connectivity.mule.api.operation.OperationResult;
import com.mulesoft.connectivity.mule.api.operation.ResultError;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.inject.Inject;

public class ConnectivityPagingProvider implements PagingProvider<Connection, Result<Object, Object>> {

  private final ExecutionContext<OperationModel> executionContext;
  private final MuleOperationSerializableModel model;
  private OperationResult<Page<?>> pageOperationResult = null;

  @Inject
  private ExpressionLanguage expressionLanguage;

  public ConnectivityPagingProvider(ExecutionContext<OperationModel> executionContext,
                                    MuleOperationSerializableModel muleOperationSerializableModel) {
    this.executionContext = executionContext;
    this.model = muleOperationSerializableModel;
  }

  @Override
  public List<Result<Object, Object>> getPage(Connection connection) {

    var provider = executionContext.getConfiguration()
        .flatMap(ConfigurationInstance::<Connection>getConnectionProvider)
        .orElseThrow();

    try {
      var connectionProvider = provider.connect();

      if (pageOperationResult == null) {
        Map<String, Object> resolvedParameters = StreamingParameterUtils
            .resolveStreamingParameters(executionContext.getParameters(), expressionLanguage);
        pageOperationResult = connectionProvider.executeOperation(model, resolvedParameters).map(p -> (Page<?>) p);

      } else {
        if (!hasNextPage(pageOperationResult)) {
          // Empty list indicates that there are no more pages
          return List.of();
        }
        pageOperationResult = connectionProvider.executeOperationNextPage(model, pageOperationResult.getValue().getNextPage()
            .orElseThrow())
            .map(p -> (Page<?>) p);
      }
      return transformToResult(pageOperationResult);
    } catch (Exception e) {
      String errorMessage = (e.getCause() != null) ? e.getCause().getMessage() : e.getMessage();
      throw new MuleRuntimeException(createStaticMessage(errorMessage), e.getCause());
    }
  }

  @Override
  public Optional<Integer> getTotalResults(Connection connection) {
    // Returns empty as per the documentation, and it's not possible to return total results from ModelInterpreter and DataWeave.
    return Optional.empty();
  }

  @Override
  public void close(Connection connection) throws MuleException {
    // ModelInterpreter executes the operation and manages the connection lifecycle. So, we don't need to close the connection
    // explicitly.
  }

  private boolean hasNextPage(OperationResult<? extends Page<?>> pageResult) {
    return pageResult.getValue().getNextPage().isPresent();
  }

  private List<Result<Object, Object>> transformToResult(OperationResult<Page<?>> pageOperationResult) {

    if (!pageOperationResult.isSuccess()) {
      ResultError errorValue = getErrorCauseIfUncheckedError(pageOperationResult.getErrorValue());
      throw new RuntimeException("Failed to extract the result of the operation: " + errorValue
          + System.lineSeparator() + "Cause: " + errorValue.getDescription() + System.lineSeparator());
    }

    List<?> items = pageOperationResult.getValue().getItems();
    List<Result<Object, Object>> resultList = new ArrayList<>();

    for (var item : items) {
      final Result.Builder<Object, Object> builder = Result.builder();
      builder.output(getRawValue((Content) item));
      builder.mediaType(getMediaType((Content) item));
      resultList.add(builder.build());
    }

    return resultList;
  }
}
