/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.source;

import static org.mule.runtime.extension.ic.internal.runtime.connection.Connection.getErrorCauseIfUncheckedError;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getMediaType;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getOutputAttributes;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getRawValue;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.PollContext;
import org.mule.runtime.extension.api.runtime.source.PollingSource;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.extension.ic.internal.runtime.connection.Connection;

import com.mulesoft.connectivity.mule.api.Content;
import com.mulesoft.connectivity.mule.api.operation.OperationResult;
import com.mulesoft.connectivity.mule.api.trigger.NextData;
import com.mulesoft.connectivity.mule.api.trigger.TriggerPage;
import com.mulesoft.connectivity.mule.persistence.model.MuleSourceSerializableModel;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectivityPollingSource extends PollingSource<Object, Object> {

  protected static final Logger LOGGER = LoggerFactory.getLogger(ConnectivityPollingSource.class);

  private final MuleSourceSerializableModel model;
  private final ConnectionProvider<?> connectionProvider;
  private final Map<String, Object> parameters;
  private Connection connection;

  public ConnectivityPollingSource(SourceFactoryContext ctx, MuleSourceSerializableModel model) {
    this.model = model;

    connectionProvider = ctx.getConnectionProvider()
        .orElseThrow(() -> new IllegalArgumentException("UC Sources require connection providers"));
    // NOTE: We might eventually want to support the case where multiple parameters on different groups have the same name
    parameters = new HashMap<>();
    ctx.getParameterization().forEachParameter((group, param, value) -> parameters.put(param.getName(), value));
  }

  @Override
  protected void doStart() throws MuleException {
    LOGGER.debug("Starting PollingSource for '{}'", getModelReference());
    connection = (Connection) connectionProvider.connect();
  }

  @Override
  protected void doStop() {
    LOGGER.debug("Stopping PollingSource '{}'", getModelReference());
  }

  @Override
  public void poll(PollContext<Object, Object> context) {
    if (context.isSourceStopping()) {
      return;
    }

    context.setWatermarkComparator((a, b) -> connection.compareWatermarks(model, a, b));
    var currentWatermark = context.getWatermark()
        .map(connection::serializeWatermark)
        .orElseGet(() -> {
          LOGGER.debug("First poll for '{}'", getModelReference());
          return connection.getInitialWatermark(model, parameters);
        });
    LOGGER.debug("Starting poll of '{}' with watermark '{}'", getModelReference(),
                 connection.deserializeWatermark(currentWatermark));

    // FIXME: If we have another kind of error we should probably throw some kind of MuleException
    // NOTE: See W-18944117 for more information
    var result = connection.executeTrigger(model, currentWatermark, parameters);
    var nextData = processResultAndGetNextData(result, context);

    while (!context.isSourceStopping() && nextData.isPresent() && nextData.get().getNextPage().isPresent()) {
      result = connection.executeTriggerNextPage(model, nextData.get());
      nextData = processResultAndGetNextData(result, context);
    }
  }

  @Override
  public void onRejectedItem(Result<Object, Object> result, SourceCallbackContext sourceCallbackContext) {
    LOGGER.debug("Item Rejected: '{}'", result.getOutput());
  }

  private Optional<NextData> processResultAndGetNextData(OperationResult<TriggerPage> result,
                                                         PollContext<Object, Object> context) {
    try {
      var page = extractSuccessfulResult(result);
      acceptAllItems(page, context);
      return Optional.of(page.getNextData());
    } catch (AccessTokenExpiredException e) {
      context.onConnectionException(new ConnectionException(e));
    }
    return Optional.empty();
  }

  private void acceptAllItems(TriggerPage page, PollContext<Object, Object> context) {
    /*
     * NOTE: We chose to use the per-item watermark instead of the per-poll watermark.
     *
     * The per-item watermark is available at `TriggerItem.getWatermark` and uses the mule PollingSource implementation to
     * deduplicate elements.
     *
     * The per-poll watermarks are available at `NextPoll.getGreatestWatermark` and `NextPoll.getLatestWatermark` if we switch to
     * those, we might enjoy common behavior between our platforms (at the expense of maybe a subpar implementation).
     */
    for (var pageItem : page.getItems()) {
      var status = context.accept(pollItem -> {
        Content content = (Content) pageItem.getValue();
        var result = Result.builder()
            .output(getRawValue(content))
            .mediaType(getMediaType(content))
            .attributes(getOutputAttributes(content))
            .build();

        pollItem
            .setResult(result)
            .setId(pageItem.getIdentity())
            .setWatermark((Serializable) pageItem.getWatermark());
      });
      if (status == PollContext.PollItemStatus.SOURCE_STOPPING) {
        return;
      }
    }
  }

  private <T> T extractSuccessfulResult(OperationResult<T> result) {
    if (result.isSuccess()) {
      return result.getValue();
    } else if (connection.isTokenExpired(result)) {
      throw new AccessTokenExpiredException();
    }
    // FIXME: This means we don't support any kind of failure, not even transient ones like connection issues
    // Handle unchecked error by getting the cause if present
    var error = getErrorCauseIfUncheckedError(result.getErrorValue());
    var errorValue = error.getValue();
    var errorDescription = error.getDescription().orElse("Unknown");
    var message =
        I18nMessageFactory.createStaticMessage("Polling operation failed: %s\n Cause: %s", errorValue, errorDescription);
    throw new MuleRuntimeException(message);
  }

  private String getModelReference() {
    return model.getModelReference().orElse("no-model-reference");
  }
}
