/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;

import org.mule.runtime.api.exception.ErrorMessageAwareException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;

/**
 * Custom exception for operation failures that implements ErrorMessageAwareException to carry the error response message built
 * using buildResult method. This exception carries the error response and implements ErrorMessageAwareException to provide rich
 * error information.
 */
public class ConnectivityException extends ModuleException implements ErrorMessageAwareException {

  private final Message message;

  /**
   * Creates a new instance.
   *
   * @param errorTypeDefinition the error type definition
   * @param errorMessage the error message
   * @param errorResult the result containing error information
   */
  public ConnectivityException(ErrorTypeDefinition<?> errorTypeDefinition,
                               String errorMessage,
                               Result<Object, Object> errorResult) {
    super(errorMessage, errorTypeDefinition);

    Message.Builder builder = Message.builder()
        .payload(getPayloadTypedValue(errorResult))
        .mediaType(getMediaType(errorResult));

    setAttributes(errorResult, builder);
    message = builder.build();
  }

  private TypedValue<Object> getPayloadTypedValue(Result<Object, Object> response) {
    if (response.getOutput() == null) {
      return TypedValue.of(null);
    }

    DataType dataType = DataType.builder()
        .type(response.getOutput().getClass())
        .mediaType(getMediaType(response))
        .build();

    return new TypedValue<>(response.getOutput(), dataType, response.getByteLength());
  }

  private void setAttributes(Result<Object, Object> errorResult, Message.Builder builder) {
    errorResult.getAttributes().ifPresent(attributes -> {
      DataType dataType = DataType.builder()
          .type(attributes.getClass())
          .mediaType(errorResult.getAttributesMediaType().orElse(APPLICATION_JAVA))
          .build();

      builder.attributes(new TypedValue<>(attributes, dataType));
    });
  }

  private MediaType getMediaType(Result<Object, Object> response) {
    return response.getMediaType().orElse(MediaType.APPLICATION_JSON);
  }

  @Override
  public Message getErrorMessage() {
    return message;
  }
}
