/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import static org.mule.runtime.extension.ic.internal.error.ConnectivityError.CLIENT_ERROR;
import static org.mule.runtime.extension.ic.internal.error.ConnectivityError.SERVER_ERROR;

import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.http.api.HttpConstants;

/**
 * Utility class for mapping status codes to standard error names. This class provides methods to resolve error names from status
 * codes.
 */
public class ErrorMapper {

  // HTTP status code ranges for error categorization
  private static final int HTTP_CLIENT_ERROR_START = 400;
  private static final int HTTP_CLIENT_ERROR_END = 500;
  private static final int HTTP_SERVER_ERROR_START = 500;
  private static final int HTTP_SERVER_ERROR_END = 600;

  /**
   * Private constructor to prevent instantiation of utility class.
   */
  private ErrorMapper() {}

  /**
   * Gets the Http Status Name based on errorKind.
   *
   * @param errorKind the error kind string
   * @return the status name of the errorKind or null
   */
  public static String getHttpStatusName(String errorKind) {
    try {
      int statusCode = getStatusCode(errorKind);
      HttpConstants.HttpStatus httpStatus = HttpConstants.HttpStatus.getStatusByCode(statusCode);
      // if statusCode doesn't match any http status code
      return (httpStatus != null) ? httpStatus.name() : null;
    } catch (NumberFormatException e) {
      // Case when status prefix is present but suffix is not a valid integer e.g status-<StringValue>
      return null;
    }
  }

  /**
   * Gets the status code from the string.
   *
   * @param statusCode the statusCode
   * @return the status code as an integer
   * @throws NumberFormatException if the error kind string does not contain a valid status code
   */
  public static int getStatusCode(String statusCode) {
    return Integer.parseInt(statusCode);
  }

  /**
   * Checks whether statusCode is a valid numeric status code.
   *
   * @param statusCode the status code to check
   * @return true if the status code is a valid numeric status code, false otherwise
   */
  public static boolean isStatusCode(String statusCode) {
    try {
      Integer.parseInt(statusCode);
      return true;
    } catch (NumberFormatException e) {
      return false;
    }
  }

  /**
   * Maps HTTP status codes to error type strings.
   *
   * @param statusCode the HTTP status code to map
   * @return the appropriate error type string
   */
  public static ErrorTypeDefinition<?> mapStatusCodeToErrorType(int statusCode) {
    if (isClientError(statusCode)) {
      return CLIENT_ERROR;
    } else if (isServerError(statusCode)) {
      return SERVER_ERROR;
    } else {
      return MuleErrors.ANY;
    }
  }

  /**
   * Checks if the status code represents a client error (4xx).
   *
   * @param statusCode the status code to check
   * @return true if it's a client error, false otherwise
   */
  private static boolean isClientError(int statusCode) {
    return statusCode >= HTTP_CLIENT_ERROR_START && statusCode < HTTP_CLIENT_ERROR_END;
  }

  /**
   * Checks if the status code represents a server error (5xx).
   *
   * @param statusCode the status code to check
   * @return true if it's a server error, false otherwise
   */
  private static boolean isServerError(int statusCode) {
    return statusCode >= HTTP_SERVER_ERROR_START && statusCode < HTTP_SERVER_ERROR_END;
  }
}
