/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.loader.property;

import org.mule.runtime.extension.api.connectivity.oauth.AuthorizationCodeGrantType;
import org.mule.runtime.extension.api.connectivity.oauth.ClientCredentialsGrantType;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthGrantType;
import org.mule.runtime.extension.api.security.CredentialsPlacement;

import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2AuthCodeAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2ClientCredentialsAuthenticationType;

/**
 * Factory class for creating instances of {@link OAuthGrantType} based on different OAuth2 authentication flows such as Client
 * Credentials and Authorization Code.
 * <p>
 * This class provides static methods to generate configured grant types using the relevant authentication type models.
 * </p>
 *
 * <p>
 * This is a utility class and cannot be instantiated.
 * </p>
 */
public class OAuthGrantTypeFactory {

  /**
   * Private constructor to prevent instantiation of this utility class.
   */
  private OAuthGrantTypeFactory() {
    // prevent instantiation
  }

  /**
   * Creates a {@link ClientCredentialsGrantType} instance using the provided {@link OAuth2ClientCredentialsAuthenticationType}.
   *
   * @param authenticationType the client credentials authentication configuration
   * @return a configured {@link OAuthGrantType} for the client credentials flow
   */
  public static OAuthGrantType createClientCredentialsGrantType(OAuth2ClientCredentialsAuthenticationType authenticationType) {
    return new ClientCredentialsGrantType(
                                          authenticationType
                                              .getTokenUrl(),
                                          "#[payload.access_token]",
                                          "#[payload.expires_in]",
                                          String
                                              .join("",
                                                    authenticationType
                                                        .getScopes()),
                                          CredentialsPlacement.BODY);
  }

  /**
   * Creates an {@link AuthorizationCodeGrantType} instance using the provided {@link OAuth2AuthCodeAuthenticationType}.
   *
   * @param authenticationType the authorization code authentication configuration
   * @return a configured {@link OAuthGrantType} for the authorization code flow
   */
  public static OAuthGrantType createAuthorizationCodeGrantType(OAuth2AuthCodeAuthenticationType authenticationType) {
    return new AuthorizationCodeGrantType(
                                          authenticationType
                                              .getTokenUrl(),
                                          authenticationType
                                              .getAuthorizationUrl(),
                                          "#[payload.access_token]",
                                          "#[payload.expires_in]",
                                          "#[payload.refresh_token]",
                                          String
                                              .join("",
                                                    authenticationType
                                                        .getScopes()),
                                          CredentialsPlacement.BODY,
                                          true);
  }
}
