/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.mule.runtime.extension.api.annotation.Extension.DEFAULT_CONFIG_DESCRIPTION;
import static org.mule.runtime.extension.api.annotation.Extension.DEFAULT_CONFIG_NAME;

import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.extension.api.loader.parser.ConfigurationModelParser;
import org.mule.runtime.extension.api.loader.parser.ConnectionProviderModelParser;
import org.mule.runtime.extension.api.loader.parser.FunctionModelParser;
import org.mule.runtime.extension.api.loader.parser.MinMuleVersionParser;
import org.mule.runtime.extension.api.loader.parser.OperationModelParser;
import org.mule.runtime.extension.api.loader.parser.ParameterGroupModelParser;
import org.mule.runtime.extension.api.loader.parser.SourceModelParser;
import org.mule.runtime.extension.api.loader.parser.StereotypeModelFactory;
import org.mule.runtime.extension.api.runtime.config.ConfigurationFactory;
import org.mule.runtime.extension.ic.internal.runtime.configuration.ConnectivityConfigurationFactory;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Configuration model parser for interpreted connectivity extensions. Transforms Mule connector model configurations into Mule
 * extension configuration models.
 */
public class ConnectivityConfigurationModelParser implements ConfigurationModelParser {

  private final MuleConnectorSerializableModel muleConnectorSerializableModel;

  public ConnectivityConfigurationModelParser(MuleConnectorSerializableModel muleConnectorSerializableModel) {
    this.muleConnectorSerializableModel = muleConnectorSerializableModel;
  }

  @Override
  public String getName() {
    return DEFAULT_CONFIG_NAME;
  }

  @Override
  public String getDescription() {
    return DEFAULT_CONFIG_DESCRIPTION;
  }

  @Override
  public ConfigurationFactory getConfigurationFactory() {
    return new ConnectivityConfigurationFactory();
  }

  @Override
  public boolean isForceNoImplicit() {
    // Returns false to allow implicit configuration creation for better user experience
    // IC extensions support automatic config instantiation when not explicitly defined
    return false;
  }

  @Override
  public List<ExternalLibraryModel> getExternalLibraryModels() {
    return List.of();
  }

  @Override
  public List<ParameterGroupModelParser> getParameterGroupParsers() {
    return List.of();
  }

  @Override
  public List<OperationModelParser> getOperationParsers() {
    return getOperationModelParsers(muleConnectorSerializableModel.getOperations());
  }

  @Override
  public List<SourceModelParser> getSourceModelParsers() {
    return muleConnectorSerializableModel.getSources().stream()
        .map(sourceModel -> new ConnectivitySourceModelParser(muleConnectorSerializableModel, sourceModel))
        .collect(Collectors.toList());
  }

  @Override
  public List<ConnectionProviderModelParser> getConnectionProviderModelParsers() {
    return getConnectionProviderModelParsers(muleConnectorSerializableModel.getConnections());
  }

  @Override
  public List<FunctionModelParser> getFunctionModelParsers() {
    return List.of();
  }

  @Override
  public Optional<DeprecationModel> getDeprecationModel() {
    return Optional.empty();
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {

    return Optional.ofNullable(DisplayModel.builder()
        .displayName(DEFAULT_CONFIG_NAME)
        .build());
  }

  @Override
  public Optional<MinMuleVersionParser> getResolvedMinMuleVersion() {
    return Optional.empty();
  }

  @Override
  public List<ModelProperty> getAdditionalModelProperties() {
    return List.of();
  }

  @Override
  public Optional<StereotypeModel> getStereotype(StereotypeModelFactory factory) {
    return Optional.empty();
  }

  private List<OperationModelParser> getOperationModelParsers(List<MuleOperationSerializableModel> operations) {
    return operations.stream()
        .map(operation -> new ConnectivityOperationModelParser(muleConnectorSerializableModel, operation, getExtensionName()))
        .collect(Collectors.toList());
  }

  private List<ConnectionProviderModelParser> getConnectionProviderModelParsers(List<MuleConnectionProviderSerializableModel> muleConnectionProviderModelList) {
    return muleConnectionProviderModelList.stream()
        .map(cpElement -> new ConnectivityConnectionProviderModelParser(muleConnectorSerializableModel, cpElement))
        .collect(Collectors.toList());
  }

  private String getExtensionName() {
    return muleConnectorSerializableModel.getName().toLowerCase();
  }
}
