/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import org.mule.runtime.extension.api.loader.parser.ErrorModelParser;

import java.util.Optional;

/**
 * Enhanced ErrorModelParser for interpreted connectivity operations that supports dynamic error hierarchy. This parser creates
 * error models based on the connectivity model's error definitions with kind.
 */
public class ConnectivityErrorModelParser implements ErrorModelParser {

  private final String type;
  private final String namespace;
  private final boolean isMuleError;
  private final boolean isSuppressed;
  private final ConnectivityErrorModelParser parent;

  /**
   * Constructor for creating an error model parser with parent hierarchy support.
   *
   * @param type the error type name
   * @param namespace the extension namespace
   * @param isMuleError whether this is a Mule error
   * @param isSuppressed whether this error should be suppressed from operation model
   * @param parent the parent error model parser (can be null for root errors)
   */
  public ConnectivityErrorModelParser(String type, String namespace, boolean isMuleError, boolean isSuppressed,
                                      ConnectivityErrorModelParser parent) {
    this.type = type;
    this.namespace = namespace;
    this.isMuleError = isMuleError;
    this.isSuppressed = isSuppressed;
    this.parent = parent;
  }

  /**
   * Constructor for backward compatibility - creates a simple error model parser.
   *
   * @param type the error type name
   * @param namespace the extension namespace
   */
  public ConnectivityErrorModelParser(String type, String namespace) {
    this(type, namespace, true, false, null);
  }

  @Override
  public String getType() {
    return type;
  }

  @Override
  public String getNamespace() {
    return namespace;
  }

  @Override
  public Optional<ErrorModelParser> getParent() {
    return Optional.ofNullable(parent);
  }

  @Override
  public boolean isMuleError() {
    return isMuleError;
  }

  /**
   * Returns whether this error should be suppressed from the operation model.
   *
   * @return true if the error should be suppressed, false otherwise
   */
  public boolean isSuppressed() {
    return isSuppressed;
  }
}
