/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.mule.runtime.api.util.JavaConstants.JAVA_VERSION_17;
import static org.mule.runtime.api.util.JavaConstants.JAVA_VERSION_21;

import static java.util.Collections.singletonList;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.notification.NotificationModel;
import org.mule.runtime.extension.api.loader.ExtensionDevelopmentFramework;
import org.mule.runtime.extension.api.loader.parser.ConfigurationModelParser;
import org.mule.runtime.extension.api.loader.parser.ConnectionProviderModelParser;
import org.mule.runtime.extension.api.loader.parser.ErrorModelParser;
import org.mule.runtime.extension.api.loader.parser.ExtensionModelParser;
import org.mule.runtime.extension.api.loader.parser.FunctionModelParser;
import org.mule.runtime.extension.api.loader.parser.LicenseModelParser;
import org.mule.runtime.extension.api.loader.parser.MinMuleVersionParser;
import org.mule.runtime.extension.api.loader.parser.OperationModelParser;
import org.mule.runtime.extension.api.loader.parser.SourceModelParser;
import org.mule.runtime.extension.api.loader.parser.XmlDslConfiguration;
import org.mule.runtime.extension.api.runtime.exception.SdkExceptionHandlerFactory;
import org.mule.runtime.extension.ic.internal.loader.ExtensionConstants;
import org.mule.sdk.api.artifact.lifecycle.ArtifactLifecycleListener;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Extension model parser for interpreted connectivity extensions. Transforms connectivity models from LinkWeave format into
 * standard Mule extension models.
 */
public class ConnectivityExtensionModelParser implements ExtensionModelParser {

  private final MuleConnectorSerializableModel muleConnectorSerializableModel;
  private final String extensionName;

  public ConnectivityExtensionModelParser(MuleConnectorSerializableModel muleConnectorSerializableModel) {
    this.muleConnectorSerializableModel = muleConnectorSerializableModel;
    extensionName = this.muleConnectorSerializableModel.getName().toLowerCase();
  }

  @Override
  public String getName() {
    return this.extensionName;
  }

  @Override
  public Category getCategory() {
    return muleConnectorSerializableModel.getCategory();
  }

  @Override
  public String getVendor() {
    return muleConnectorSerializableModel.getVendor();
  }

  @Override
  public List<ConfigurationModelParser> getConfigurationParsers() {
    return singletonList(new ConnectivityConfigurationModelParser(muleConnectorSerializableModel));
  }

  @Override
  public List<OperationModelParser> getOperationModelParsers() {
    return List.of();
  }

  @Override
  public List<SourceModelParser> getSourceModelParsers() {
    return List.of();
  }

  @Override
  public List<ConnectionProviderModelParser> getConnectionProviderModelParsers() {
    return List.of();
  }

  @Override
  public List<FunctionModelParser> getFunctionModelParsers() {
    return List.of();
  }

  @Override
  public List<ErrorModelParser> getErrorModelParsers() {
    return List.of();
  }

  @Override
  public LicenseModelParser getLicensingParser() {
    return new ConnectivityLicenseModelParser(true, true);
  }

  @Override
  public List<ExternalLibraryModel> getExternalLibraryModels() {
    return List.of();
  }

  @Override
  public Optional<SdkExceptionHandlerFactory> getExceptionHandlerFactory() {
    return Optional.empty();
  }

  @Override
  public Optional<DeprecationModel> getDeprecationModel() {
    return Optional.empty();
  }

  @Override
  public Optional<XmlDslConfiguration> getXmlDslConfiguration() {
    return Optional
        .of(new XmlDslConfiguration(extensionName, ExtensionConstants.EXTENSION_NAMESPACE_PREFIX + extensionName));
  }

  @Override
  public List<MetadataType> getExportedTypes() {
    return List.of();
  }

  @Override
  public List<String> getExportedResources() {
    return List.of();
  }

  @Override
  public List<MetadataType> getImportedTypes() {
    return List.of();
  }

  @Override
  public List<String> getPrivilegedExportedArtifacts() {
    return List.of();
  }

  @Override
  public List<String> getPrivilegedExportedPackages() {
    return List.of();
  }

  @Override
  public Map<MetadataType, List<MetadataType>> getSubTypes() {
    return Map.of();
  }

  @Override
  public List<NotificationModel> getNotificationModels() {
    return List.of();
  }

  @Override
  public String getNamespace() {
    return ExtensionConstants.EXTENSION_NAMESPACE_PREFIX
        + extensionName;
  }


  @Override
  public Optional<MinMuleVersionParser> getResolvedMinMuleVersion() {
    return Optional.empty();
  }

  @Override
  public ExtensionDevelopmentFramework getDevelopmentFramework() {
    return ExtensionDevelopmentFramework.IC_SDK;
  }

  @Override
  public Optional<Class<? extends ArtifactLifecycleListener>> getArtifactLifecycleListenerClass() {
    return Optional.empty();
  }

  @Override
  public Set<String> getSupportedJavaVersions() {
    return Set.of(JAVA_VERSION_17, JAVA_VERSION_21);
  }

  @Override
  public List<ModelProperty> getAdditionalModelProperties() {
    return List.of();
  }
}

