/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.extension.api.annotation.source.SourceClusterSupport;
import org.mule.runtime.extension.api.loader.parser.BackPressureModeParser;
import org.mule.runtime.extension.api.loader.parser.OutputModelParser;
import org.mule.runtime.extension.api.loader.parser.SourceModelParser;
import org.mule.runtime.extension.api.runtime.source.BackPressureMode;
import org.mule.runtime.extension.api.runtime.source.SdkSourceFactory;
import org.mule.runtime.extension.api.runtime.source.Source;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.extension.ic.internal.parser.utils.AnnotationUtils;
import org.mule.runtime.extension.ic.internal.runtime.source.ConnectivityPollingSource;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleSourceSerializableModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectivitySourceModelParser extends ConnectivityExecutableModelParser<MuleSourceSerializableModel>
    implements SourceModelParser {

  protected static final Logger LOGGER = LoggerFactory.getLogger(ConnectivitySourceModelParser.class);

  public ConnectivitySourceModelParser(MuleConnectorSerializableModel connectorModel, MuleSourceSerializableModel sourceModel) {
    super(connectorModel, sourceModel);

    if (!(sourceModel.getInputType() instanceof ObjectType)) {
      throw new IllegalArgumentException("The input type for mule sources cannot be a non-object type");
    }
  }

  @Override
  public OutputModelParser getOutputType() {
    return new ConnectivityOutputModelParser(model.getOutputType(), false);
  }

  @Override
  public OutputModelParser getAttributesOutputType() {
    // FIXME: Which attributes should the polling items have?
    return new ConnectivityOutputModelParser(BaseTypeBuilder.create(MetadataFormat.JAVA).voidType().build(), false);
  }

  @Override
  public Optional<SourceCallbackModelParser> getOnSuccessCallbackParser() {
    return Optional.empty();
  }

  @Override
  public Optional<SourceCallbackModelParser> getOnErrorCallbackParser() {
    return Optional.empty();
  }

  @Override
  public Optional<SourceCallbackModelParser> getOnTerminateCallbackParser() {
    return Optional.empty();
  }

  @Override
  public Optional<SourceCallbackModelParser> getOnBackPressureCallbackParser() {
    return Optional.empty();
  }

  @Override
  public boolean isPolling() {
    return true;
  }

  @Override
  public Optional<SdkSourceFactory> getSourceFactory() {
    Optional<SdkSourceFactory> maybeSourceFactory = model.getModelReference()
        .map(modelReference -> new SdkSourceFactory() {

          @Override
          public Either<org.mule.sdk.api.runtime.source.Source, Source> createMessageSource() {
            throw new UnsupportedOperationException("UC does not support building mule sources without a SourceFactoryContext");
          }

          @Override
          public Source<?, ?> createMessageSource(SourceFactoryContext ctx) {
            return new ConnectivityPollingSource(ctx, model);
          }
        });

    if (maybeSourceFactory.isEmpty()) {
      LOGGER.error("Skipping {} since it lacks a model reference", model.getName());
    }
    return maybeSourceFactory;
  }

  @Override
  public boolean emitsResponse() {
    return false;
  }

  @Override
  public boolean supportsStreaming() {
    return false;
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    var displayModel = DisplayModel.builder()
        .displayName(model.getDisplayName())
        .build();
    return Optional.of(displayModel);
  }

  @Override
  public Optional<BackPressureModeParser> getBackPressureModeParser() {
    // For now we are only supporting WAIT, if the need arises the set of supported modes can be expanded.
    // See W-19158082 for more info.
    return Optional.of(new BackPressureModeParser() {

      @Override
      public BackPressureMode getDefaultMode() {
        return BackPressureMode.WAIT;
      }

      @Override
      public Set<BackPressureMode> getSupportedModes() {
        return Set.of(BackPressureMode.WAIT);
      }
    });
  }

  @Override
  public Optional<SourceClusterSupport> getSourceClusterSupport() {
    return Optional.of(SourceClusterSupport.DEFAULT_ALL_NODES);
  }

  @Override
  public Set<String> getSemanticTerms() {
    return AnnotationUtils.getSemanticTerms(this.model.getInputType().getAnnotations());
  }

  @Override
  public List<ModelProperty> getAdditionalModelProperties() {
    return List.of();
  }
}
