/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser.utils;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.SemanticTermsTypeAnnotation;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

public class AnnotationUtils {

  private AnnotationUtils() {}

  /**
   * Finds the value of a specific annotation type from a set of annotations.
   *
   * @param <T> The type of annotation to find
   * @param annotations The set of annotations to search through
   * @param annotationClass The class of the annotation to find
   * @return Optional containing the annotation value if found, empty otherwise
   */
  public static <T extends TypeAnnotation> Optional<T> findAnnotation(Set<TypeAnnotation> annotations,
                                                                      Class<T> annotationClass) {
    return annotations.stream()
        .filter(annotationClass::isInstance)
        .map(annotationClass::cast)
        .findFirst();
  }

  /**
   * Extracts semantic terms from a set of annotations.
   *
   * @param annotations The set of annotations to search through
   * @return Set of semantic terms found in the annotations
   */
  public static Set<String> getSemanticTerms(Set<TypeAnnotation> annotations) {
    return annotations.stream()
        .filter(SemanticTermsTypeAnnotation.class::isInstance)
        .map(SemanticTermsTypeAnnotation.class::cast)
        .map(SemanticTermsTypeAnnotation::getSemanticTerms)
        .flatMap(Set::stream)
        .collect(Collectors.toSet());
  }
}
