/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.connection;

import static org.mule.runtime.api.connection.ConnectionValidationResult.failure;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.ic.internal.error.ConnectivityError;
import org.mule.runtime.extension.ic.internal.runtime.interpreter.ModelInterpreterProvider;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;

import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;

import java.util.Map;

import jakarta.inject.Inject;

public abstract class BaseConnectivityConnectionProvider
    implements CachedConnectionProvider<Connection>, Initialisable, Startable, Stoppable {

  @Inject
  protected ModelInterpreterProvider modelInterpreterProvider;

  @Inject
  protected HttpService httpService;

  protected TlsContextFactory tlsContext;

  protected HttpClient httpClient;

  private final MuleConnectionProviderSerializableModel connectionProviderModel;
  private static final String ERROR_MESSAGE = "Error testing connection";
  private static final String ERROR_DETAILS = "No additional info";
  private static final String HTTP_CLIENT_NAME = "universal-connectivity-%s-http-client";

  /**
   * Enhanced constructor that sets up validation with the provided models.
   *
   * @param connectionProviderModel the connection provider model
   */
  public BaseConnectivityConnectionProvider(MuleConnectionProviderSerializableModel connectionProviderModel) {
    this.connectionProviderModel = connectionProviderModel;
  }

  @Override
  public ConnectionValidationResult validate(Connection connection) {
    try {
      var result = connection.testConnection();

      if (result.isValid()) {
        return success();
      } else {
        String errorMessage = isEmpty(result.getMessage()) ? ERROR_MESSAGE : result.getMessage();
        String errorDetails = isEmpty(result.getException().getMessage()) ? ERROR_DETAILS : result.getException().getMessage();
        return ConnectionValidationResult.failure(errorMessage, new RuntimeException(errorMessage + ": " + errorDetails));
      }
    } catch (RuntimeException e) {
      return failure(ERROR_MESSAGE, new ModuleException(e.getMessage(), ConnectivityError.CONNECTIVITY, e));
    }
  }

  private boolean isEmpty(String str) {
    return str == null || str.isEmpty();
  }

  @Override
  public final void initialise() throws InitialisationException {
    // initialize the TLS
    initialiseIfNeeded(tlsContext);
  }

  @Override
  public void start() {
    startHttpClient();
  }

  private void startHttpClient() {
    HttpClientConfiguration.Builder configuration = new HttpClientConfiguration.Builder()
        .setTlsContextFactory(tlsContext)
        .setName(String.format(HTTP_CLIENT_NAME, connectionProviderModel.getAuthenticationTypeName()));

    if (httpService == null) {
      throw new RuntimeException("Failed to setup Mule Runtime HttpService");
    }
    httpClient = httpService.getClientFactory().create(configuration.build());
    httpClient.start();
  }

  @Override
  public void stop() {
    httpClient.stop();
  }

  /**
   * Creates a new {@link Connection} instance with the configured parameters.
   *
   * @return a new connection
   */
  @Override
  public Connection connect() {
    return new Connection(getParameters(), connectionProviderModel, modelInterpreterProvider.getMuleConnectivityService(),
                          httpClient);
  }

  @Override
  public void disconnect(Connection connection) {
    // ModelInterpreter executes the test connection and manages the connection lifecycle. So, we don't need to disconnect
    // the connection.
  }

  /**
   * Returns the configured parameters for this connection provider.
   *
   * @return A map filled with <code>(paramName, paramValue)</code> entries
   */
  public abstract Map<String, Object> getParameters();

}
