/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.connection;

import static java.util.function.Predicate.not;

import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.http.api.client.HttpClient;

import com.mulesoft.connectivity.mule.api.MuleConnectivityService;
import com.mulesoft.connectivity.mule.api.Page;
import com.mulesoft.connectivity.mule.api.ServiceConfiguration;
import com.mulesoft.connectivity.mule.api.operation.OperationResult;
import com.mulesoft.connectivity.mule.api.operation.ResultError;
import com.mulesoft.connectivity.mule.api.trigger.NextData;
import com.mulesoft.connectivity.mule.api.trigger.TriggerPage;
import com.mulesoft.connectivity.mule.api.valueprovider.ProvidedValue;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleSourceSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleValueProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.SerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.HttpAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;

import java.io.Serializable;
import java.util.Map;
import java.util.Optional;

/**
 * Responsible for a Connection on the LW framework. Holds enough information to forward execution requests to
 * {@link MuleConnectivityService}.
 */
public class Connection {

  private static final String ACCESS_TOKEN_EXPIRED_ERROR_KIND = "401";

  private final Map<String, Object> params;

  final private MuleConnectionProviderSerializableModel model;
  private final ServiceConfiguration config;
  private final MuleConnectivityService muleConnectivityService;

  /**
   * Constructs a new Connection with the specified parameters and ModelInterpreterProvider.
   *
   * @param params the parameters for the connection
   * @param model the connection model
   * @param muleConnectivityService the service that performs execution of UC components
   * @param httpClient the http client
   */
  public Connection(Map<String, Object> params, MuleConnectionProviderSerializableModel model,
                    MuleConnectivityService muleConnectivityService, HttpClient httpClient) {
    this.params = params;
    this.model = model;
    this.muleConnectivityService = muleConnectivityService;
    this.config = new ServiceConfiguration(Map.of(ServiceConfiguration.CLIENT_TYPE.HTTPCLIENT.name(), httpClient));
  }

  public ConnectionValidationResult testConnection() {
    String testConnectionSelector = getModelReference(model.getTestConnection());
    String connectionProviderSelector = getModelReference(model);
    return muleConnectivityService.testConnectivity(testConnectionSelector, connectionProviderSelector, params, config);
  }

  public OperationResult<?> executeOperation(MuleOperationSerializableModel operationModel, Map<String, Object> operationParams) {
    String connectionProviderSelector = getModelReference(model);
    String operationModelReference = getModelReference(operationModel);
    return muleConnectivityService.executeOperation(connectionProviderSelector, params, operationModelReference, operationParams,
                                                    operationModel.isPaginated(), config);
  }


  public OperationResult<?> executeOperationNextPage(MuleOperationSerializableModel operationModel, Object operationParams) {
    String connectionProviderSelector = getModelReference(model);
    String operationModelReference = getModelReference(operationModel);
    return muleConnectivityService.executeOperationNextPage(connectionProviderSelector, params, operationModelReference,
                                                            operationParams,
                                                            config);
  }

  public OperationResult<Page<ProvidedValue>> executeValueProvider(MuleValueProviderSerializableModel valueProviderModel,
                                                                   Map<String, Object> valueProviderParams) {
    String connectionProviderSelector = getModelReference(model);
    String valueProviderModelReference = getModelReference(valueProviderModel);
    return muleConnectivityService.executeValueProvider(connectionProviderSelector, params, valueProviderModelReference,
                                                        valueProviderParams,
                                                        valueProviderModel.isPaginated(), config);
  }

  public OperationResult<Page<ProvidedValue>> executeValueProviderNextPage(MuleValueProviderSerializableModel valueProviderModel,
                                                                           Object nextPageData) {
    String connectionProviderSelector = getModelReference(model);
    String valueProviderModelReference = getModelReference(valueProviderModel);
    return muleConnectivityService.executeValueProviderNextPage(connectionProviderSelector, params,
                                                                valueProviderModelReference, nextPageData, config);
  }

  private static String getModelReference(SerializableModel model) {
    return model.getModelReference().filter(not(String::isEmpty))
        .orElseThrow(() -> new IllegalStateException("Model must have a valid selector set as model reference"));
  }


  public HttpAuthenticationType getConnectionType() {
    return model.getAuthenticationType();
  }

  public int compareWatermarks(MuleSourceSerializableModel sourceModel, Serializable a, Serializable b) {
    // NOTE: Java comparators are the reverse to IC watermark comparators. Therefore, we just flip the arguments.
    return muleConnectivityService.executeTriggerCompareWatermark(getModelReference(sourceModel), b, a, config);
  }

  public Serializable serializeWatermark(Serializable watermark) {
    return muleConnectivityService.serializeWatermarkObject(watermark, config);
  }

  public Serializable getInitialWatermark(MuleSourceSerializableModel sourceModel, Map<String, Object> sourceParams) {
    return muleConnectivityService.getInitialWatermark(getModelReference(model), params,
                                                       getModelReference(sourceModel), sourceParams, config);
  }

  public Object deserializeWatermark(Serializable currentWatermark) {
    return muleConnectivityService.deserializeWatermarkObject(currentWatermark, config);
  }

  public OperationResult<TriggerPage> executeTrigger(MuleSourceSerializableModel sourceModel, Serializable watermark,
                                                     Map<String, Object> sourceParams) {
    return muleConnectivityService.executeTrigger(getModelReference(model), params, getModelReference(sourceModel), watermark,
                                                  sourceParams, config);
  }

  public OperationResult<TriggerPage> executeTriggerNextPage(MuleSourceSerializableModel sourceModel, NextData nextData) {
    return muleConnectivityService.executeTriggerNextPage(getModelReference(model), params, getModelReference(sourceModel),
                                                          nextData, config);
  }

  /**
   * Determines whether the given {@link OperationResult} indicates that the access token has expired and a refresh is required.
   *
   * @param result the result of the operation, potentially containing an error value
   * @return if the status code matches {@code ACCESS_TOKEN_EXPIRED_ERROR_KIND} and the token is expired and a refresh is needed
   */
  public boolean isTokenExpired(OperationResult<?> result) {
    HttpAuthenticationType type = model.getAuthenticationType();
    if (type != null && type.getType() == HttpAuthenticationType.Type.oauth2) {
      String errorKind = result.getErrorValue().getKind();
      return errorKind.equalsIgnoreCase(ACCESS_TOKEN_EXPIRED_ERROR_KIND);
    }
    return false;
  }

  /**
   * Retrieves the cause of an error if it's an unchecked error.
   *
   * This method checks if the given error is of kind "UNCHECKED" and has a cause. If both conditions are met, it returns the
   * cause of the error. Otherwise, it returns the original error.
   *
   * @param errorValue The ResultError to examine
   * @return The cause of the error if it's an unchecked error with a cause, otherwise the original error
   */
  public static ResultError getErrorCauseIfUncheckedError(ResultError errorValue) {
    Optional<ResultError> resultErrorCause = errorValue.getCause();
    if (errorValue.getKind().equalsIgnoreCase(ResultError.KIND_UNCHECKED) && resultErrorCause.isPresent()) {
      errorValue = resultErrorCause.get();
    }
    return errorValue;
  }
}
