/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.connection;

import org.mule.runtime.extension.api.runtime.connectivity.ConnectionProviderFactory;

import com.mulesoft.connectivity.mule.persistence.model.connection.HttpAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2AuthenticationType;

import java.util.Optional;
import java.util.function.Supplier;

/**
 * Responsible for creating instances of BaseConnectivityConnectionProvider.
 */
public class ConnectivityConnectionProviderFactory implements ConnectionProviderFactory<Connection> {

  private final Class<? extends BaseConnectivityConnectionProvider> selectedProviderClass;
  private final Supplier<? extends BaseConnectivityConnectionProvider> selectedProvider;

  /**
   * Initializes the factory with the given connection provider model and test connection model.
   *
   * @param serializedProvider the connection provider model
   */
  public ConnectivityConnectionProviderFactory(MuleConnectionProviderSerializableModel serializedProvider) {
    var authenticationType = serializedProvider.getAuthenticationType();

    selectedProviderClass = switch (authenticationType.getType()) {
      case bearer -> BearerConnectivityConnectionProvider.class;
      case basic -> BasicConnectivityConnectionProvider.class;
      case apiKey -> APIKeyConnectivityConnectionProvider.class;
      case oauth2 -> getOAuth2ProviderClass(authenticationType);
      default -> throw new IllegalArgumentException("Unsupported authentication type: " + authenticationType);
    };

    selectedProvider = switch (authenticationType.getType()) {
      case bearer -> () -> new BearerConnectivityConnectionProvider(serializedProvider);
      case basic -> () -> new BasicConnectivityConnectionProvider(serializedProvider);
      case apiKey -> () -> new APIKeyConnectivityConnectionProvider(serializedProvider);
      case oauth2 -> () -> getOAuth2Provider(authenticationType, serializedProvider);
      default -> throw new IllegalArgumentException("Unsupported authentication type: " + authenticationType);
    };
  }

  @Override
  public BaseConnectivityConnectionProvider newInstance() {
    return selectedProvider.get();
  }

  @Override
  public Class<? extends BaseConnectivityConnectionProvider> getObjectType() {
    return selectedProviderClass;
  }

  private Class<? extends BaseConnectivityConnectionProvider> getOAuth2ProviderClass(HttpAuthenticationType httpAuthType) {
    OAuth2AuthenticationType.GrantType grantType = getGrantType(httpAuthType);
    return switch (grantType) {
      case authorizationCode -> OAuth2AuthorizationCodeConnectionProvider.class;
      case clientCredentials -> OAuth2ClientCredentialsConnectionProvider.class;
    };
  }

  private BaseConnectivityConnectionProvider getOAuth2Provider(HttpAuthenticationType httpAuthType,
                                                               MuleConnectionProviderSerializableModel serializedProvider) {
    OAuth2AuthenticationType.GrantType grantType = getGrantType(httpAuthType);
    return switch (grantType) {
      case authorizationCode -> new OAuth2AuthorizationCodeConnectionProvider(serializedProvider);
      case clientCredentials -> new OAuth2ClientCredentialsConnectionProvider(serializedProvider);
    };
  }

  private OAuth2AuthenticationType.GrantType getGrantType(HttpAuthenticationType httpAuthType) {
    Optional<String> subType = httpAuthType.getSubType();
    if (subType.isPresent()) {
      return OAuth2AuthenticationType.GrantType.fromString(subType.get());
    } else {
      throw new IllegalArgumentException("SubType cannot be null for OAuth authentication");
    }
  }
}
