/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.operation;

import static org.mule.runtime.extension.ic.internal.runtime.connection.Connection.getErrorCauseIfUncheckedError;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getMediaType;
import static org.mule.runtime.extension.ic.internal.utils.ContentExtractorUtils.getOutputAttributes;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.operation.CompletableComponentExecutor;
import org.mule.runtime.extension.api.runtime.operation.ExecutionContext;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.ic.internal.error.ConnectivityError;
import org.mule.runtime.extension.ic.internal.error.ConnectivityException;
import org.mule.runtime.extension.ic.internal.error.ErrorTypeDefinitionFactory;
import org.mule.runtime.extension.ic.internal.runtime.connection.Connection;
import org.mule.runtime.extension.ic.internal.runtime.operation.pagination.ConnectivityPagingProvider;
import org.mule.runtime.extension.ic.internal.utils.StreamingParameterUtils;

import com.mulesoft.connectivity.mule.api.Content;
import com.mulesoft.connectivity.mule.api.operation.OperationResult;
import com.mulesoft.connectivity.mule.api.operation.ResultError;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;

import java.util.Map;
import java.util.Set;

import javax.inject.Inject;

public class OperationExecutor implements CompletableComponentExecutor<OperationModel> {

  private final MuleOperationSerializableModel model;

  @Inject
  private ExpressionLanguage expressionLanguage;

  public OperationExecutor(MuleOperationSerializableModel model) {
    this.model = model;
  }

  /**
   * Executes an operation using the provided execution context and returns the result through the callback. The method handles
   * connection establishment, operation execution, and result processing.
   *
   * @param executionContext The context containing operation parameters and configuration
   * @param executorCallback The callback to handle operation completion or errors
   */
  @Override
  public void execute(ExecutionContext<OperationModel> executionContext, ExecutorCallback executorCallback) {

    try {
      var provider = executionContext.getConfiguration()
          .flatMap(ConfigurationInstance::<Connection>getConnectionProvider)
          .orElseThrow();
      var connection = provider.connect();

      if (model.isPaginated()) {
        executorCallback.complete(new ConnectivityPagingProvider(executionContext, model));
      } else {
        Map<String, Object> resolvedParameters = StreamingParameterUtils
            .resolveStreamingParameters(executionContext.getParameters(), expressionLanguage);

        var result = connection.executeOperation(model, resolvedParameters);
        if (result.isSuccess()) {
          Content content = (Content) result.getValue();
          executorCallback.complete(buildResult(content));
        } else if (connection.isTokenExpired(result)) {
          executorCallback.error(new AccessTokenExpiredException());
        } else {
          // Handle operation failure with proper error handling
          handleOperationFailure(executionContext.getExtensionModel().getName(), result,
                                 executionContext.getComponentModel().getErrorModels(), executorCallback);
        }
      }
    } catch (ModuleException e) {
      executorCallback.error(e);
    } catch (Exception e) {
      executorCallback.error(new ModuleException(e.getMessage(), ConnectivityError.CONNECTIVITY, e));
    }
  }

  private static Result<Object, Object> buildResult(Content content) {
    Result.Builder<Object, Object> builder = Result.builder();

    MediaType mediaType = getMediaType(content);

    builder.output("");
    builder.mediaType(MediaType.ANY);

    content.getRawValue().ifPresent(rawVal -> {
      builder.output(rawVal);
      builder.mediaType(mediaType);
    });

    builder.attributes(getOutputAttributes(content));
    builder.attributesMediaType(MediaType.APPLICATION_JAVA);

    return builder.build();
  }


  /**
   * Handles operation failure by creating appropriate error types and throwing the correct exception. Uses
   * ErrorTypeDefinitionFactory to determine the error type and creates a ConnectivityException(Module Exception) with proper
   * error message.
   *
   * @param extensionName the ExtensionName
   * @param result the operation result containing error information
   * @param errorModels the list of errorModels
   * @param executorCallback the callback to handle the error
   */
  private void handleOperationFailure(String extensionName, OperationResult<?> result, Set<ErrorModel> errorModels,
                                      CompletableComponentExecutor.ExecutorCallback executorCallback) {

    // Handle unchecked error by getting the cause if present
    ResultError errorValue = getErrorCauseIfUncheckedError(result.getErrorValue());

    // Extract error information
    String kind = errorValue.getKind();

    // Create error type definition using the factory
    ErrorTypeDefinition<?> errorTypeDefinition =
        ErrorTypeDefinitionFactory.createErrorTypeDefinition(extensionName, kind, errorModels);

    String errorMessage = buildErrorMessage(errorValue, errorTypeDefinition);

    // Build error result using the existing buildResult method
    Result<Object, Object> errorResult = errorValue.getContent()
        .map(OperationExecutor::buildResult)
        .orElse(Result.builder().build());

    // throw ConnectivityException with proper error type definition
    executorCallback.error(new ConnectivityException(errorTypeDefinition, errorMessage, errorResult));
  }

  /**
   * Builds a descriptive error message using the error description from ResultError or falls back to the error type definition
   * when no description is available.
   *
   * @param error the result error containing error information
   * @param errorTypeDefinition the error type definition
   * @return the formatted error message
   */
  private String buildErrorMessage(ResultError error, ErrorTypeDefinition<?> errorTypeDefinition) {
    // Use error description if available, otherwise fall back to error type definition
    String fallbackMessage = errorTypeDefinition.getType();

    String errorMessage = error.getDescription()
        .filter(desc -> !desc.trim().isEmpty())
        .orElse(fallbackMessage);

    return "Error occurred:  " + errorMessage;
  }
}
