/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.valueProvider;

import static org.mule.runtime.extension.api.values.ValueResolvingException.MISSING_REQUIRED_PARAMETERS;
import static org.mule.runtime.extension.ic.internal.runtime.connection.Connection.getErrorCauseIfUncheckedError;

import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactoryContext;
import org.mule.runtime.extension.api.values.ValueBuilder;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;
import org.mule.runtime.extension.ic.internal.runtime.connection.Connection;

import com.mulesoft.connectivity.mule.api.Page;
import com.mulesoft.connectivity.mule.api.operation.OperationResult;
import com.mulesoft.connectivity.mule.api.operation.ResultError;
import com.mulesoft.connectivity.mule.api.valueprovider.ProvidedValue;
import com.mulesoft.connectivity.mule.persistence.model.MuleValueProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedContextReferenceVariable;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedObjectFieldSelector;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderArgument;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedTypeReferenceExpression;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Executes value providers for the interpreted connectivity extension. Dynamically resolves parameter values from injected fields
 * and executes the value provider through the model interpreter to return selectable values.
 *
 * Enhanced with sophisticated nested path resolution using patterns from ObjectTypeHandler and BindingSupport, including
 * type-aware path traversal, context-aware resolution.
 */
public class ValueProviderExecutor implements ValueProvider {

  public ValueProviderFactoryContext valueProviderFactoryContext;

  private final MuleValueProviderSerializableModel model;
  private final List<SerializedProviderArgument> providerArguments;

  private static final String PROCESSING_FAILURE = "PROCESSING_FAILURE";

  public ValueProviderExecutor(ValueProviderFactoryContext valueProviderFactoryContext,
                               MuleValueProviderSerializableModel model,
                               List<SerializedProviderArgument> providerArguments) {
    this.valueProviderFactoryContext = valueProviderFactoryContext;
    this.model = model;
    this.providerArguments = providerArguments;
  }

  @Override
  public Set<Value> resolve() throws ValueResolvingException {
    try {
      Map<String, Object> parameters = fetchParamValues(model);
      Connection connection = getConnection();

      List<ProvidedValue> allProvidedValues = fetchAllPages(connection, parameters);
      return convertToValues(allProvidedValues);

    } catch (ValueResolvingException e) {
      throw e;
    } catch (Exception e) {
      throw new ValueResolvingException("Failed to resolve values: " + e.getMessage(), PROCESSING_FAILURE);
    }
  }

  private Connection getConnection() {
    return (Connection) this.valueProviderFactoryContext.getConnectionSupplier()
        .orElseThrow()
        .get();
  }

  private List<ProvidedValue> fetchAllPages(Connection connection, Map<String, Object> parameters)
      throws ValueResolvingException {
    List<ProvidedValue> allValues = new ArrayList<>();
    OperationResult<Page<ProvidedValue>> pageResult = connection.executeValueProvider(model, parameters);

    // Add first page results
    if (pageResult.isSuccess()) {
      allValues.addAll(pageResult.getValue().getItems());
    }

    // Fetch subsequent pages
    while (pageResult.isSuccess() && hasNextPage(pageResult.getValue())) {
      pageResult = fetchNextPage(connection, pageResult.getValue().getNextPage().orElseThrow());
      if (pageResult.isSuccess()) {
        allValues.addAll(pageResult.getValue().getItems());
      }
    }

    // Check for final failure
    if (!pageResult.isSuccess()) {
      // Handle unchecked error by getting the cause if present
      ResultError errorValue = getErrorCauseIfUncheckedError(pageResult.getErrorValue());
      throw new ValueResolvingException("Failed to execute operation: " + errorValue.getValue(),
                                        PROCESSING_FAILURE);
    }

    return allValues;
  }

  private boolean hasNextPage(Page<ProvidedValue> page) {
    return page.getNextPage().isPresent();
  }

  private OperationResult<Page<ProvidedValue>> fetchNextPage(Connection connection, Serializable nextPageToken) {
    return connection.executeValueProviderNextPage(model, nextPageToken);
  }

  private Set<Value> convertToValues(List<ProvidedValue> providedValues) {
    return providedValues.stream()
        .map(this::createValue)
        .collect(Collectors.toSet());
  }

  private Value createValue(ProvidedValue providedValue) {
    return ValueBuilder.newValue(providedValue.getValue().toString())
        .withDisplayName(providedValue.getLabel())
        .build();
  }

  private Map<String, Object> fetchParamValues(MuleValueProviderSerializableModel valueProviderSerializableModel)
      throws ValueResolvingException {
    Map<String, Object> parameters = new HashMap<>();
    List<String> missingRequiredParameters = new ArrayList<>();

    for (SerializedProviderArgument argument : providerArguments) {
      String parameterName = getParameterNameFromSelector(argument.getParameterSelector());
      Object parameterValue = resolveParameterValueFromInputSelector(argument.getInputSelector());
      if (parameterValue != null && !parameterValue.equals("")) {
        parameters.put(parameterName, parameterValue);
      } else if (isParameterRequired(valueProviderSerializableModel, parameterName)) {
        missingRequiredParameters.add(getExtractionExpression(argument.getInputSelector()));
      }
    }

    validateRequiredParameters(missingRequiredParameters);
    return parameters;
  }

  private void validateRequiredParameters(List<String> missingRequiredParameters) throws ValueResolvingException {
    if (!missingRequiredParameters.isEmpty()) {
      throw new ValueResolvingException(
                                        "Unable to retrieve values. There are missing required parameters for the resolution: "
                                            + missingRequiredParameters,
                                        MISSING_REQUIRED_PARAMETERS);
    }
  }

  private String getExtractionExpression(SerializedTypeReferenceExpression inputSelector) {
    if (inputSelector instanceof SerializedContextReferenceVariable contextVar) {
      return contextVar.getName();
    } else if (inputSelector instanceof SerializedObjectFieldSelector objectFieldSelector) {
      return String.join(".", objectFieldSelector.getPath());
    } else {
      throw new RuntimeException("Unknown type reference expression: " + inputSelector.getClass());
    }
  }

  private boolean isParameterRequired(MuleValueProviderSerializableModel valueProviderModel, String parameterName) {
    for (ObjectFieldType field : ((DefaultObjectType) valueProviderModel.getInputType()).getFields()) {
      String paramName = field.getKey().getName().toString();
      if (field.isRequired() && parameterName.equals(paramName)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Extracts the parameter name from a parameter selector. For simple selectors like "state", this returns "state". For complex
   * selectors, it takes the first path element.
   *
   * @param parameterSelector param selection
   * @return actual param name part of operation input type
   */
  private String getParameterNameFromSelector(SerializedObjectFieldSelector parameterSelector) {
    String[] path = parameterSelector.getPath();
    if (path.length == 0) {
      throw new RuntimeException("Invalid parameter Selector");
    }
    return path[0];
  }

  /**
   * Resolves the actual parameter value based on the input selector. This handles both relative and absolute selectors, and
   * context variables. Enhanced with sophisticated nested path resolution using patterns from ObjectTypeHandler and
   * BindingSupport.
   *
   * @param inputSelector inputSelector
   * @return returns param value from operation input type
   */
  private Object resolveParameterValueFromInputSelector(SerializedTypeReferenceExpression inputSelector) {
    if (inputSelector instanceof SerializedContextReferenceVariable contextVar) {
      return resolveContextVariable(contextVar);
    } else if (inputSelector instanceof SerializedObjectFieldSelector objectFieldSelector) {
      return resolveObjectFieldSelector(objectFieldSelector);
    } else {
      throw new RuntimeException("Unknown type reference expression: " + inputSelector.getClass());
    }
  }

  private Object resolveContextVariable(SerializedContextReferenceVariable contextVar) {
    return valueProviderFactoryContext.getComponentParameterization()
        .getParameter(contextVar.getName(), contextVar.getName());
  }

  private Object resolveObjectFieldSelector(SerializedObjectFieldSelector objectFieldSelector) {
    String[] path = objectFieldSelector.getPath();
    validatePath(path);

    Object currentValue = getRootParameterValue(path[0]);
    return traversePath(currentValue, path);
  }

  private void validatePath(String[] path) {
    if (path.length == 0) {
      throw new RuntimeException("Invalid expression: empty path");
    }
  }

  private Object getRootParameterValue(String rootParameter) {
    try {
      return valueProviderFactoryContext.getComponentParameterization().getParameter(rootParameter, rootParameter);
    } catch (Exception ignored) {
      // Return null for missing parameters - validation will handle required parameters
      return null;
    }
  }

  private Object traversePath(Object currentValue, String[] path) {
    for (int i = 1; i < path.length; i++) {
      currentValue = traversePathElement(currentValue, path[i]);
    }
    return currentValue;
  }

  /**
   * Traverses a single path element using type-aware logic similar to ObjectFieldSelector.evaluateOn(). Supports complex
   * DataWeave types including Maps, Objects, and nested structures.
   *
   * @param currentValue the current value being traversed.
   * @param pathElement the path element to traverse.
   * @return the value at the specified path element.
   */
  private Object traversePathElement(Object currentValue, String pathElement) {
    if (currentValue == null) {
      return null;
    }

    if (currentValue instanceof Map) {
      @SuppressWarnings("unchecked")
      Map<String, Object> map = (Map<String, Object>) currentValue;
      return map.get(pathElement);
    }
    throw new RuntimeException("Input value is not of type map: " + currentValue);
  }
}
