/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.utils;

import static org.mule.runtime.extension.ic.internal.utils.Constants.STATUS_CODE;

import org.mule.runtime.api.metadata.MediaType;

import com.mulesoft.connectivity.mule.api.Content;

import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class ContentExtractorUtils {

  private ContentExtractorUtils() {}

  public static InputStream getRawValue(Content content) {
    return content.getRawValue()
        .orElseThrow(() -> new RuntimeException("Content is empty"));
  }

  /**
   * Returns a map of attributes extracted from the provided content by the transport, or null if no status code is present.
   *
   * @param content the content from which to extract the attributes.
   * @return a map of attributes.
   */
  public static Map<String, Object> getOutputAttributes(Content content) {
    Map<String, Object> attributes = new HashMap<>();
    var transportAttributes = content.getTransportAttributes().orElseGet(Collections::emptyMap);
    attributes.put(STATUS_CODE, transportAttributes.get(STATUS_CODE));
    return attributes;
  }

  /**
   * Returns the media type for the given content, or {@link MediaType#APPLICATION_JSON} if no content type is present.
   *
   * @param content the content from which to extract the media type.
   * @return the media type.
   */
  public static MediaType getMediaType(Content content) {
    return content.getContentType()
        .filter(contentType -> !contentType.isBlank())
        .map(MediaType::parse)
        .orElse(MediaType.APPLICATION_JSON);
  }
}
