/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.error;

import static org.mule.runtime.extension.ic.internal.error.ErrorMapper.getHttpStatusName;
import static org.mule.runtime.extension.ic.internal.error.ErrorMapper.getStatusCode;
import static org.mule.runtime.extension.ic.internal.error.ErrorMapper.isStatusCode;
import static org.mule.runtime.extension.ic.internal.error.ErrorMapper.mapStatusCodeToErrorType;
import static org.mule.runtime.extension.ic.internal.utils.StringUtils.isNullOrEmpty;

import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.extension.ic.internal.parser.ConnectivityErrorModelParser;

import com.mulesoft.connectivity.mule.api.operation.ResultError;
import com.mulesoft.connectivity.mule.persistence.model.MuleErrorSerializableModel;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Factory class for creating ConnectivityErrorModelParser instances with dynamic error hierarchy.
 * <p>
 * This factory processes error models from the Mule connectivity model and creates appropriate error model parsers with
 * well-defined parent-child relationships.
 * </p>
 */
public class ConnectivityErrorModelParserFactory {

  private static final boolean IS_NOT_MULE_ERROR = false;
  private static final boolean IS_NOT_SUPPRESSED = false;

  private ConnectivityErrorModelParserFactory() {}

  /**
   * Creates ConnectivityErrorModelParser instances with dynamic hierarchy from error models.
   * <p>
   * Processes each error model and creates appropriate error parsers with parent-child relationships. Ensures no duplicate error
   * types are returned by using the error type name as the uniqueness key.
   * </p>
   *
   * @param errorModels list of error models from the connectivity model (can be null or empty)
   * @param extensionNamespace the extension namespace (must not be null)
   * @return a list of unique ConnectivityErrorModelParser instances, or empty list if no valid error models provided
   */
  public static List<ConnectivityErrorModelParser> createErrorModelParsers(
                                                                           List<MuleErrorSerializableModel> errorModels,
                                                                           String extensionNamespace) {

    if (errorModels == null || errorModels.isEmpty()) {
      return List.of();
    }

    // Create a list to store the ConnectivityErrorModelParsers
    List<ConnectivityErrorModelParser> errorModelParsers = new ArrayList<>();

    // Use a set to track seen error types for O(1) duplicate detection
    Set<String> seenErrorTypes = new LinkedHashSet<>();

    // Default predefined errorModels will always be there for each operation
    errorModelParsers.addAll(getAllPreDefinedErrorModelParser(extensionNamespace));

    // Process each error model to create the appropriate error parser
    for (MuleErrorSerializableModel errorModel : errorModels) {
      String errorKind = errorModel.getKind();

      // Create the error parser for the current error model
      ConnectivityErrorModelParser errorParser = createErrorModelParser(errorKind, extensionNamespace);

      if (errorParser != null && seenErrorTypes.add(errorParser.getType())) {
        errorModelParsers.add(errorParser);
      }
    }

    return errorModelParsers;
  }

  /**
   * Creates an error parser for a single error model based on its kind.
   * <p>
   * Determines the appropriate error hierarchy based on the error model's kind. The method handles various error kind formats and
   * creates proper parent-child relationships in the error hierarchy.
   * </p>
   *
   * @param errorKind the error kind (must not be null)
   * @param extensionNamespace the extension namespace (must not be null)
   * @return the created error parser for the given error model, or root ANY parser if kind is null/empty
   */
  public static ConnectivityErrorModelParser createErrorModelParser(
                                                                    String errorKind,
                                                                    String extensionNamespace) {

    // Note: statusCode will be either Empty, statusCode or a customValue

    // Case 1: ErrorKind will be __UNCHECKED
    // Predefined errorModels should be defined at the caller end
    if (isNullOrEmpty(errorKind) || errorKind.equalsIgnoreCase(ResultError.KIND_UNCHECKED)) {
      // Error kinds will be __UNCHECKED for generic status patterns like:
      // status-default, status-unexpected, status-unauthorized, status-4xx, status-5xx
      return null;
    }

    // Case 2: ErrorKind will be a statusCode
    if (isStatusCode(errorKind)) {
      // ErrorKind will be the statusCode
      String statusName = getHttpStatusName(errorKind);

      // statusName can be null of statusCode doesn't match to HttpStatus codes
      if (statusName == null) {
        // Error Hierarchy will be either CLIENT_ERROR -> ANY, SERVER_ERROR -> ANY, or ANY based on statusCode(errorKind)
        return getDefaultErrorModelParser(errorKind, extensionNamespace);
      } else {
        // Error Hierarchy will be either STATUSNAME -> CLIENT_ERROR/SERVER_ERROR/ANY(based on statusCode) -> ANY
        return createErrorModelParserFromResolvedName(statusName, errorKind, extensionNamespace);
      }
    } else {
      // Case 3: ErrorKind will be a customValue
      // Error Hierarchy will be either CUSTOMVALUE -> ANY
      return createErrorModelParserFromResolvedName(errorKind, null, extensionNamespace);
    }
  }

  /**
   * Creates an error model parser from a resolved error type name.
   *
   * @param errorTypeName the resolved error type name
   * @param errorKind the errorKind
   * @param extensionNamespace the extension namespace
   * @return the created error model parser
   */
  private static ConnectivityErrorModelParser createErrorModelParserFromResolvedName(String errorTypeName,
                                                                                     String errorKind,
                                                                                     String extensionNamespace) {
    // Check if it matches to any predefined connectivity Error
    ConnectivityError connectivityError = ConnectivityError.fromString(errorTypeName);

    if (connectivityError == null) {
      // If it doesn't match to any predefined connectivity Error
      ConnectivityErrorModelParser parentParser = getDefaultErrorModelParser(errorKind, extensionNamespace);
      return createErrorModelParser(errorTypeName, extensionNamespace, IS_NOT_MULE_ERROR, parentParser);
    } else {
      // If it matches to any predefined connectivity Error
      return createErrorModelParserChain(connectivityError, extensionNamespace);
    }
  }

  /**
   * Creates a default error model parser based on the error kind.
   * <p>
   * For numeric status codes, categorizes them into CLIENT_ERROR, SERVER_ERROR, or ANY. For non-numeric error kinds (custom
   * values), defaults to ANY error type.
   * </p>
   *
   * @param statusCode the error kind without status prefix
   * @param extensionNamespace the extension namespace
   * @return the default error model parser
   */
  public static ConnectivityErrorModelParser getDefaultErrorModelParser(String statusCode,
                                                                        String extensionNamespace) {
    ErrorTypeDefinition<?> errorType;

    errorType = isStatusCode(statusCode) ? mapStatusCodeToErrorType(getStatusCode(statusCode)) : MuleErrors.ANY;

    return createErrorModelParserChain(errorType, extensionNamespace);
  }

  /**
   * Creates an error model parser with all predefined errors.
   *
   * @param extensionNamespace the extension namespace
   * @return the List of error model parser
   */
  private static List<ConnectivityErrorModelParser> getAllPreDefinedErrorModelParser(String extensionNamespace) {
    List<ConnectivityErrorModelParser> errorModelParsers = new ArrayList<>();
    for (ErrorTypeDefinition<ConnectivityError> errorTypeDefinition : ConnectivityError.getErrorTypeList()) {
      errorModelParsers.add(createErrorModelParserChain(errorTypeDefinition, extensionNamespace));
    }
    return errorModelParsers;
  }

  /**
   * Creates a ConnectivityErrorModelParser with the specified parameters.
   *
   * @param errorType the error type name
   * @param namespace the namespace for the error
   * @param isMuleError whether this is a Mule framework error
   * @param parent the parent error parser in the hierarchy
   * @return a new ConnectivityErrorModelParser instance
   */
  private static ConnectivityErrorModelParser createErrorModelParser(
                                                                     String errorType, String namespace, boolean isMuleError,
                                                                     ConnectivityErrorModelParser parent) {

    return new ConnectivityErrorModelParser(errorType, namespace, isMuleError, IS_NOT_SUPPRESSED, parent);
  }

  /**
   * Creates a ConnectivityErrorModelParser chain with parent-child hierarchy recursively from errorTypeDefinition.
   *
   * @param errorTypeDefinition the error type definition
   * @param extensionNameSpace the extension namespace
   * @return a ConnectivityErrorModelParser with hierarchy
   */
  private static ConnectivityErrorModelParser createErrorModelParserChain(ErrorTypeDefinition<?> errorTypeDefinition,
                                                                          String extensionNameSpace) {
    // Get the parent error type definition if it exists
    Optional<ErrorTypeDefinition<? extends Enum<?>>> parentErrorTypeDefinition = errorTypeDefinition.getParent();

    ConnectivityErrorModelParser parentParser = null;

    // If there's a parent, recursively create the parent chain first
    if (parentErrorTypeDefinition.isPresent()) {
      @SuppressWarnings("unchecked")
      ErrorTypeDefinition<ConnectivityError> parentConnectivityErrorDef =
          (ErrorTypeDefinition<ConnectivityError>) parentErrorTypeDefinition.get();
      parentParser = createErrorModelParserChain(parentConnectivityErrorDef, extensionNameSpace);
    }

    // Create the error model parser for the current error type with its parent
    boolean isMuleError = errorTypeDefinition instanceof org.mule.runtime.extension.api.error.MuleErrors;
    extensionNameSpace = isMuleError ? ConnectivityErrorConstants.MULE_NAMESPACE : extensionNameSpace;
    return createErrorModelParser(errorTypeDefinition.getType(), extensionNameSpace, isMuleError, parentParser);
  }
}
