/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;

import static org.apache.commons.lang.StringUtils.EMPTY;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.connection.ConnectionManagementType;
import org.mule.runtime.api.meta.model.deprecated.DeprecationModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthGrantType;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthModelProperty;
import org.mule.runtime.extension.api.loader.parser.ConnectionProviderModelParser;
import org.mule.runtime.extension.api.loader.parser.MinMuleVersionParser;
import org.mule.runtime.extension.api.loader.parser.ParameterGroupModelParser;
import org.mule.runtime.extension.api.loader.parser.StereotypeModelFactory;
import org.mule.runtime.extension.api.runtime.connectivity.ConnectionProviderFactory;
import org.mule.runtime.extension.ic.internal.loader.property.OAuthGrantTypeFactory;
import org.mule.runtime.extension.ic.internal.parser.utils.AnnotationUtils;
import org.mule.runtime.extension.ic.internal.parser.utils.ModelParserUtils;
import org.mule.runtime.extension.ic.internal.runtime.connection.ConnectivityConnectionProviderFactory;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.HttpAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.MuleConnectionProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2AuthCodeAuthenticationType;
import com.mulesoft.connectivity.mule.persistence.model.connection.oauth.OAuth2ClientCredentialsAuthenticationType;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Connection provider model parser for interpreted connectivity extensions. Transforms Mule connection provider models into Mule
 * extension connection provider models.
 */
public class ConnectivityConnectionProviderModelParser implements ConnectionProviderModelParser {

  private final MuleConnectorSerializableModel muleConnectorSerializableModel;
  private final MuleConnectionProviderSerializableModel muleConnectionProviderSerializableModel;
  private final Set<TypeAnnotation> annotations;

  public ConnectivityConnectionProviderModelParser(MuleConnectorSerializableModel muleConnectorSerializableModel,
                                                   MuleConnectionProviderSerializableModel muleConnectionProviderSerializableModel) {
    this.muleConnectorSerializableModel = muleConnectorSerializableModel;
    this.muleConnectionProviderSerializableModel = muleConnectionProviderSerializableModel;
    this.annotations = muleConnectionProviderSerializableModel.getInputType().getAnnotations();
  }


  @Override
  public String getName() {
    return muleConnectionProviderSerializableModel.getName();
  }

  @Override
  public String getDescription() {
    return AnnotationUtils.findAnnotation(annotations, DescriptionAnnotation.class)
        .map(DescriptionAnnotation::getValue)
        .orElse(EMPTY);
  }

  @Override
  public List<ParameterGroupModelParser> getParameterGroupModelParsers() {
    // Omitting the 'accessToken' field (in paramsToOmit) from connection configuration UI as the token
    // is managed internally by the connector and exposing it to the user is not required.
    return ModelParserUtils
        .getParameterGroupParser(muleConnectorSerializableModel, muleConnectionProviderSerializableModel.getInputType(),
                                 new HashMap<>(),
                                 DEFAULT_GROUP_NAME, true,
                                 paramsToOmit(muleConnectionProviderSerializableModel));
  }

  @Override
  public List<ExternalLibraryModel> getExternalLibraryModels() {
    return List.of();
  }

  @Override
  public ConnectionManagementType getConnectionManagementType() {
    return ConnectionManagementType.NONE;
  }

  @Override
  public Optional<ConnectionProviderFactory<?>> getConnectionProviderFactory() {
    return Optional.of(new ConnectivityConnectionProviderFactory(muleConnectionProviderSerializableModel));
  }

  @Override
  public boolean supportsConnectivityTesting() {
    return muleConnectionProviderSerializableModel.getTestConnection() != null;
  }


  @Override
  public boolean supportsXa() {
    return false;
  }

  @Override
  public boolean isExcludedFromConnectivitySchema() {
    // Returns false because connection providers should be included in the connectivity schema
    // to enable proper tooling support and configuration validation in Studio and other tools.
    return false;
  }

  @Override
  public Optional<OAuthModelProperty> getOAuthModelProperty() {
    List<OAuthGrantType> grantTypes = new LinkedList();
    if (muleConnectionProviderSerializableModel
        .getAuthenticationType() instanceof OAuth2ClientCredentialsAuthenticationType authTypeCc) {
      grantTypes.add(OAuthGrantTypeFactory.createClientCredentialsGrantType(authTypeCc));
    } else if (muleConnectionProviderSerializableModel
        .getAuthenticationType() instanceof OAuth2AuthCodeAuthenticationType authTypeAc) {
      grantTypes.add(OAuthGrantTypeFactory.createAuthorizationCodeGrantType(authTypeAc));
    }
    return grantTypes.isEmpty() ? Optional.empty() : Optional.of(new OAuthModelProperty(grantTypes));
  }

  @Override
  public List<ModelProperty> getAdditionalModelProperties() {
    return List.of();
  }

  @Override
  public Optional<DeprecationModel> getDeprecationModel() {
    return Optional.empty();
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    Optional<String> displayName = AnnotationUtils.findAnnotation(annotations, LabelAnnotation.class)
        .map(LabelAnnotation::getValue);
    Optional<String> description =
        AnnotationUtils.findAnnotation(annotations, DescriptionAnnotation.class)
            .map(DescriptionAnnotation::getValue);

    if (displayName.isPresent() || description.isPresent()) {
      DisplayModel.DisplayModelBuilder builder = DisplayModel.builder();
      displayName.ifPresent(builder::displayName);
      description.ifPresent(builder::summary);
      return Optional.of(builder.build());
    }

    return Optional.empty();
  }

  @Override
  public Optional<MinMuleVersionParser> getResolvedMinMuleVersion() {
    return Optional.empty();
  }

  @Override
  public Set<String> getSemanticTerms() {
    return AnnotationUtils.getSemanticTerms(annotations);
  }

  @Override
  public Optional<StereotypeModel> getStereotype(StereotypeModelFactory factory) {
    return Optional.empty();
  }

  /**
   * Determines a list of parameter names to omit based on the authentication type of the given Mule connection provider model.
   *
   * <p>
   * Currently, it omits the {@code accessToken} field if the authentication type name starts with {@code oauth2}.
   * </p>
   *
   * @param muleConnectionProviderSerializableModel the connection provider model to inspect
   * @return a list of parameter names that should be omitted from input metadata
   */
  private List<String> paramsToOmit(MuleConnectionProviderSerializableModel muleConnectionProviderSerializableModel) {
    final String INPUT_FIELD_ACCESS_TOKEN = "accessToken";
    List<String> paramsToBeRemoved = new ArrayList<>();
    if (muleConnectionProviderSerializableModel.getAuthenticationTypeName()
        .startsWith(HttpAuthenticationType.Type.oauth2.name())) {
      paramsToBeRemoved.add(INPUT_FIELD_ACCESS_TOKEN);
    }
    return paramsToBeRemoved;
  }
}
