/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.mule.runtime.extension.ic.internal.utils.Constants.STATUS_CODE;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.operation.ExecutionType;
import org.mule.runtime.extension.api.loader.parser.ErrorModelParser;
import org.mule.runtime.extension.api.loader.parser.NestedChainModelParser;
import org.mule.runtime.extension.api.loader.parser.NestedRouteModelParser;
import org.mule.runtime.extension.api.loader.parser.OperationModelParser;
import org.mule.runtime.extension.api.loader.parser.OutputModelParser;
import org.mule.runtime.extension.api.loader.parser.metadata.InputResolverModelParser;
import org.mule.runtime.extension.api.loader.parser.metadata.RoutesChainInputTypesResolverModelParser;
import org.mule.runtime.extension.api.loader.parser.metadata.ScopeChainInputTypeResolverModelParser;
import org.mule.runtime.extension.api.runtime.operation.CompletableComponentExecutorFactory;
import org.mule.runtime.extension.ic.internal.error.ConnectivityErrorModelParserFactory;
import org.mule.runtime.extension.ic.internal.parser.utils.AnnotationUtils;
import org.mule.runtime.extension.ic.internal.runtime.operation.OperationExecutorFactory;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleErrorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleOperationSerializableModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

public class ConnectivityOperationModelParser extends ConnectivityExecutableModelParser<MuleOperationSerializableModel>
    implements OperationModelParser {

  private final Set<TypeAnnotation> annotations;
  private final String extensionNamespace;

  public ConnectivityOperationModelParser(MuleConnectorSerializableModel muleConnectorSerializableModel,
                                          MuleOperationSerializableModel muleOperationSerializableModel,
                                          String extensionNamespace) {
    super(muleConnectorSerializableModel, muleOperationSerializableModel);
    this.extensionNamespace = extensionNamespace;
    this.annotations = muleOperationSerializableModel.getInputType().getAnnotations();
  }

  @Override
  public OutputModelParser getOutputType() {
    MetadataType originalOutputType = model.getOutputType();
    return new ConnectivityOutputModelParser(originalOutputType, false);
  }

  @Override
  public OutputModelParser getAttributesOutputType() {
    MetadataType attributesType =
        model.isPaginated() ? BaseTypeBuilder.create(MetadataFormat.JAVA).voidType().build() : getAttributesType();
    return new ConnectivityOutputModelParser(attributesType, false);
  }

  @Override
  public List<NestedRouteModelParser> getNestedRouteParsers() {
    return List.of();
  }

  @Override
  public Optional<CompletableComponentExecutorFactory<?>> getExecutor() {
    return Optional.of(new OperationExecutorFactory(model));
  }

  @Override
  public Optional<NestedChainModelParser> getNestedChainParser() {
    return Optional.empty();
  }

  @Override
  public boolean isBlocking() {
    return true;
  }

  @Override
  public boolean isScope() {
    return false;
  }

  @Override
  public boolean isRouter() {
    return false;
  }

  @Override
  public boolean supportsStreaming() {
    return this.model.isPaginated();
  }

  @Override
  public boolean isAutoPaging() {
    return this.model.isPaginated();
  }

  @Override
  public boolean hasStreamingConfiguration() {
    return supportsStreaming();
  }

  @Override
  public boolean hasTransactionalAction() {
    return false;
  }

  @Override
  public boolean hasReconnectionStrategy() {
    return false;
  }

  @Override
  public boolean propagatesConnectivityError() {
    return false;
  }

  @Override
  public Optional<ExecutionType> getExecutionType() {
    return Optional.empty();
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    return Optional.ofNullable(DisplayModel.builder().displayName(this.model.getDisplayName()).build());
  }

  @Override
  public List<ErrorModelParser> getErrorModelParsers() {
    return getErrorModelParsersList();
  }

  @Override
  public List<InputResolverModelParser> getInputResolverModelParsers() {
    return List.of();
  }

  @Override
  public Optional<ScopeChainInputTypeResolverModelParser> getScopeChainInputTypeResolverModelParser() {
    return Optional.empty();
  }

  @Override
  public Optional<RoutesChainInputTypesResolverModelParser> getRoutesChainInputTypesResolverModelParser() {
    return Optional.empty();
  }

  @Override
  public Set<String> getSemanticTerms() {
    return AnnotationUtils.getSemanticTerms(this.annotations);
  }

  private List<ErrorModelParser> getErrorModelParsersList() {
    // Get error models from the operation
    List<MuleErrorSerializableModel> errorModels = model.getErrorModelList();

    // Use the factory to create error model parsers with dynamic hierarchy
    List<ConnectivityErrorModelParser> errorParsers =
        ConnectivityErrorModelParserFactory.createErrorModelParsers(errorModels, this.extensionNamespace.toUpperCase());

    return errorParsers.stream()
        .map(ErrorModelParser.class::cast)
        .collect(java.util.stream.Collectors.toList());
  }

  /**
   * Gets the attributes type, which is a java object containing a string field named 'statusCode'.
   *
   * @return The attributes type.
   */
  private MetadataType getAttributesType() {
    var objectBuilder = new BaseTypeBuilder(MetadataFormat.JAVA).objectType();
    objectBuilder
        .addField()
        .key(STATUS_CODE)
        .value(getStringType());
    return objectBuilder.build();
  }

  private MetadataType getStringType() {
    return new BaseTypeBuilder(MetadataFormat.JAVA).stringType().build();
  }

}
