/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.parser;

import static org.apache.commons.lang.StringUtils.EMPTY;

import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.extension.api.loader.parser.ParameterModelParser;
import org.mule.runtime.extension.ic.internal.parser.utils.AnnotationUtils;

import com.mulesoft.connectivity.mule.persistence.model.MuleConnectorSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.MuleValueProviderSerializableModel;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedObjectFieldSelector;
import com.mulesoft.connectivity.mule.persistence.model.provider.SerializedProviderReferences;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Parameter group model parser for interpreted connectivity extensions. Transforms LinkWeave type models into Mule extension
 * parameter group models.
 */
public class ConnectivityParameterGroupParser implements ParameterGroupModelParserWithDefaults {

  private final MetadataType inputMetadataType;
  private final Set<TypeAnnotation> annotations;
  private final String parameterGroupName;
  private final List<String> paramsToOmit;
  private final boolean isConnectionProvider;
  private final MuleConnectorSerializableModel muleConnectorSerializableModel;
  private final Map<SerializedObjectFieldSelector, SerializedProviderReferences> inputResolvedProviders;

  public ConnectivityParameterGroupParser(MuleConnectorSerializableModel muleConnectorSerializableModel,
                                          MetadataType metadataType,
                                          Map<SerializedObjectFieldSelector, SerializedProviderReferences> inputResolvedProviders,
                                          String parameterGroupName, boolean isConnectionProvider,
                                          List<String> paramsToOmit) {
    this.muleConnectorSerializableModel = muleConnectorSerializableModel;
    this.inputMetadataType = metadataType;
    this.inputResolvedProviders = inputResolvedProviders;
    this.annotations = inputMetadataType.getAnnotations();
    this.parameterGroupName = parameterGroupName;
    this.isConnectionProvider = isConnectionProvider;
    this.paramsToOmit = paramsToOmit;
  }

  @Override
  public String getName() {
    return this.parameterGroupName;
  }

  @Override
  public String getDescription() {
    return EMPTY;
  }

  @Override
  public List<ParameterModelParser> getParameterParsers() {
    if (inputMetadataType instanceof ObjectType) {
      List<ParameterModelParser> parsers =
          ((ObjectType) inputMetadataType).getFields().stream()
              .filter(param -> !paramsToOmit.contains(param.getKey().getName().getLocalPart()))
              .map(param -> new ConnectivityParameterModelParser(param, getValueProviderModel(param),
                                                                 fetchProviderReferences(param)))
              .collect(Collectors.toList());
      if (isConnectionProvider) {
        parsers.add(new TlsParser());
      }
      return parsers;
    }
    return isConnectionProvider ? List.of(new TlsParser()) : List.of();
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    return Optional.of(DisplayModel.builder()
        .displayName(AnnotationUtils.findAnnotation(annotations, LabelAnnotation.class)
            .map(LabelAnnotation::getValue)
            .orElse(EMPTY))
        .build());
  }


  private MuleValueProviderSerializableModel getValueProviderModel(ObjectFieldType param) {

    String valueProviderReference = getValueProviderReference(fetchProviderReferences(param));
    if (valueProviderReference.isEmpty()) {
      return null;
    }
    return muleConnectorSerializableModel.getValueProvider(valueProviderReference);
  }

  private SerializedProviderReferences fetchProviderReferences(ObjectFieldType field) {
    for (SerializedObjectFieldSelector selector : inputResolvedProviders.keySet()) {
      if (selector.getPath()[0].equals(field.getKey().getName().getLocalPart())) {
        return inputResolvedProviders.get(selector);
      }
    }
    return null;
  }

  public String getValueProviderReference(SerializedProviderReferences providerReferences) {
    return providerReferences != null && providerReferences.getValueProviderReference() != null
        ? providerReferences.getValueProviderReference().getName()
        : "";
  }
}
