/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.runtime.configuration;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.extension.api.runtime.config.ConfigurationFactory;

import java.lang.reflect.InvocationTargetException;
import java.util.function.Supplier;

/**
 * Represents a configuration factory for creating configuration instances.
 */
public class ConnectivityConfigurationFactory implements ConfigurationFactory {

  /**
   * Supplier for the generated configuration class.
   */
  private final Supplier<Class<?>> configClass;

  /**
   * Constructs an instance of the {@link ConnectivityConfigurationFactory}.
   *
   */
  public ConnectivityConfigurationFactory() {
    this.configClass = new LazyValue<>(() -> ConnectivityConfigurationFactory.class);
  }

  /**
   * Creates a new instance of the dynamically generated configuration class.
   *
   * @return Instance of the generated configuration class
   */
  @Override
  public Object newInstance() {
    try {
      return this.configClass.get().getDeclaredConstructor().newInstance();
    } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
        | NoSuchMethodException | SecurityException e) {
      throw new MuleRuntimeException(e);
    }
  }

  /**
   * Retrieves the type of the generated configuration class.
   *
   * @return Type of the generated configuration class
   */
  @Override
  public Class<?> getObjectType() {
    return configClass.get();
  }
}
