/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.runtime.extension.ic.internal.utils;

import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.Cursor;
import org.mule.runtime.api.streaming.CursorProvider;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.ic.internal.error.ConnectivityError;

import java.io.InputStream;
import java.util.Map;

/**
 * Utility class for handling streaming parameters in Mule operations. This class provides methods to resolve CursorProvider and
 * InputStream parameters by consuming their content and converting them to structured data.
 */
public class StreamingParameterUtils {

  private static final DataType JAVA_OBJECT = DataType.builder().type(Object.class).mediaType(APPLICATION_JAVA).build();
  private static final String PAYLOAD = "payload";
  private static final String EXPRESSION = "#[read(payload)]";
  private static final String UNSUPPORTED_MIME_TYPE_ERROR_MESSAGE = "Unsupported MIME type for the given payload";

  private StreamingParameterUtils() {}

  /**
   * Resolves streaming parameters by handling CursorProvider and InputStream parameters. This method processes parameters and
   * converts streaming resources to consumable data.
   *
   * @param parameters the original parameters map
   * @param expressionLanguage the expression language instance to use for resolving expressions
   * @return a new map with resolved streaming parameters
   */
  public static Map<String, Object> resolveStreamingParameters(Map<String, Object> parameters,
                                                               ExpressionLanguage expressionLanguage) {
    Map<String, Object> resolvedParameters = new java.util.HashMap<>(parameters);

    for (Map.Entry<String, Object> entry : resolvedParameters.entrySet()) {
      try {
        Object value = entry.getValue();
        if (value instanceof CursorProvider<?> cursorProvider) {
          Cursor cursor = cursorProvider.openCursor();
          InputStream inputStream = (InputStream) cursor;
          entry.setValue(convert(inputStream, expressionLanguage).getValue());
        } else if (value instanceof InputStream) {
          entry.setValue(convert((InputStream) value, expressionLanguage).getValue());
        }
      } catch (ModuleException e) {
        throw e;
      } catch (Exception e) {
        throw new ModuleException(e.getMessage(), ConnectivityError.CONNECTIVITY, e);
      }
      // Else handles non-streamable entries(eg: object) by returning them without modification.
    }

    return resolvedParameters;
  }

  @SuppressWarnings("unchecked")
  private static <T> TypedValue<T> convert(InputStream inputStream, ExpressionLanguage expressionLanguage) {
    TypedValue<Object> payload = TypedValue.of(inputStream);
    BindingContext context = BindingContext.builder().addBinding(PAYLOAD, payload).build();
    try {
      return (TypedValue<T>) expressionLanguage.evaluate(EXPRESSION, JAVA_OBJECT, context);
    } catch (Exception e) {
      throw new ModuleException(UNSUPPORTED_MIME_TYPE_ERROR_MESSAGE, MuleErrors.EXPRESSION);
    }
  }
}
