/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.capability.xml.schema.model;

import org.jetbrains.annotations.NotNull;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.invocation.InvocationOnMock;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.mockito.Mockito.mock;

/**
 * Some tests to exercise getters and setters on the interface classes
 */
class ModelElementPropertyTestCase {

  private static final Class<?>[] CLASSES = {
      Annotated.class,
      Annotation.class,
      Any.class,
      Appinfo.class,
      Attribute.class,
      AttributeGroup.class,
      BuiltBy.class,
      ComplexContent.class,
      ComplexType.class,
      Documentation.class,
      Element.class,
      ExtensionType.class,
      Facet.class,
      org.mule.runtime.module.extension.internal.capability.xml.schema.model.Field.class,
      Group.class,
      Include.class,
      Keybase.class,
      Keyref.class,
      org.mule.runtime.module.extension.internal.capability.xml.schema.model.List.class,
      NamedGroup.class,
      Notation.class,
      OpenAttrs.class,
      Redefine.class,
      Restriction.class,
      RestrictionType.class,
      Schema.class,
      Selector.class,
      SimpleContent.class,
      SimpleType.class,
      Union.class,
      Wildcard.class
  };

  private static final Pattern GETTER_PATTERN = Pattern.compile("^(is|get)(.*)");
  private static final Class<?>[] ENUM_CLASSES = {
      DerivationControl.class,
      TypeDerivationControl.class,
      ReducedDerivationControl.class,
      FormChoice.class
  };

  /**
   * Check that the getter returns the value set by the setter, if one is present, or at least not a null if there is no setter.
   *
   * @param testName just the class name of the class under test
   * @param getter   the getter method (getX or isX for booleans) we're testing
   * @throws InvocationTargetException bubble up the injustice of the world
   * @throws IllegalAccessException    bubble up the injustice of the world
   * @throws InstantiationException    bubble up the injustice of the world
   */
  @ParameterizedTest(name = "[{index}]{0}")
  @MethodSource("propertyMethods")
  void accessorsAccess(String testName, Method getter)
      throws InvocationTargetException, IllegalAccessException, InstantiationException {
    final Class<?> classUnderTest = getter.getDeclaringClass();
    final Object instance = getInstance(classUnderTest);
    final Matcher matcher = GETTER_PATTERN.matcher(getter.getName());
    if (!matcher.matches()) {
      throw new IllegalArgumentException("Getter method " + getter.getName() + " did not match expected pattern");
    }
    final String propertyName = matcher.group(2);
    final Optional<Method> setter = Arrays.stream(classUnderTest.getDeclaredMethods())
        .filter(m -> m.getName().equals("set" + propertyName) && m.getParameterCount() == 1)
        .findFirst();
    Object setValue = null;
    if (setter.isPresent()) {
      final Method method = setter.get();
      final Class<?> parameterType = method.getParameterTypes()[0];
      setValue = getParam(parameterType);
      method.invoke(instance, setValue);
    }
    final Object result = getter.invoke(instance);

    if (setValue != null) {
      assertThat(testName + ":" + getter.getName(), result, is(setValue));
    } else {
      assertThat(testName + ":" + getter.getName(), result, is(notNullValue()));
    }
  }

  @ParameterizedTest(name = "[{index}]{0}")
  @MethodSource("enumMethods")
  void enumValuesMap(String testName, Method fromMethod, Object testValue, Object expectedValue)
      throws InvocationTargetException, IllegalAccessException {
    final Class<?> declaringClass = fromMethod.getDeclaringClass();
    final Object result = fromMethod.invoke(null, testValue);

    assertThat(declaringClass.getSimpleName() + ":" + testName, result, is(expectedValue));
  }

  /**
   * Take the list of classes we're going to test and get all the getter methods on them.
   *
   * @return arguments that include a test name and the method to invoke for the test.
   */
  static List<Arguments> propertyMethods() {
    return Arrays.stream(CLASSES)
        .flatMap(c -> Arrays.stream(c.getDeclaredMethods()))
        .filter(m -> m.getName().matches("^(get|is).*") && m.getParameterCount() == 0)
        .map(m -> Arguments.of(m.getDeclaringClass().getSimpleName(), m))
        .toList();
  }

  /**
   * Take the list of enum types and convert it to a list of arguments for each value in each enum. The arguments are the test
   * name, the 'fromValue' method, the value to convert for the test, and the expected enum value.
   *
   * @return The arguments for tests...
   */
  static List<Arguments> enumMethods() {
    return Arrays.stream(ENUM_CLASSES)
        .flatMap(c -> Arrays.stream(c.getDeclaredMethods()))
        .filter(m -> m.getName().matches("^from.*") && m.getParameterCount() == 1)
        .flatMap(m -> getEnumValues(m).map(ev -> Arguments.of(m.getName(), m, getValue(ev), ev)))
        .toList();
  }

  /**
   * Return an instance of the class being tested. Some of them are abstract classes and so must use a mock that just calls the
   * real methods (formerly known as a spy)
   *
   * @param classUnderTest the class we want to instantiate
   * @return an object of the given class for testing.
   * @throws InstantiationException    If we couldn't instantiate the object
   * @throws IllegalAccessException    We shouldn't instantiate the object
   * @throws InvocationTargetException We daren't instantiate the object
   */
  @NotNull
  private static Object getInstance(Class<?> classUnderTest)
      throws InstantiationException, IllegalAccessException, InvocationTargetException {
    if (Modifier.isAbstract(classUnderTest.getModifiers())) {
      return mock(classUnderTest, InvocationOnMock::callRealMethod);
    } else {
      return Arrays.stream(classUnderTest.getDeclaredConstructors()).filter(c -> c.getParameterCount() == 0)
          .peek(c -> c.setAccessible(true))
          .findFirst()
          .orElseThrow(() -> new IllegalArgumentException("No zero arg constructor for " + classUnderTest.getSimpleName()))
          .newInstance();
    }
  }

  /**
   * Create a parameter for invoking a method.
   *
   * @param parameterType the type we need for the parameter
   * @return an object suitable for the purpose (might be a mock or a real object)
   */
  private static Object getParam(Class<?> parameterType) {
    if (String.class.equals(parameterType)) {
      return "Marmoset!";
    } else if (Boolean.class.equals(parameterType)) {
      return Boolean.TRUE;
    }
    return mock(parameterType);
  }

  /**
   * Invoke the method to return the 'value' of an enum type. This may be a string, or another enum...
   *
   * @param enumValue the enum value we want the 'value' value for.
   * @return the result of enumValue.value()
   * @throws IllegalArgumentException if we can't get the value for any reason.
   */
  private static Object getValue(Enum<?> enumValue) {
    try {
      final Class<?> declaringClass = enumValue.getDeclaringClass();
      Optional<Method> method = Arrays.stream(declaringClass.getDeclaredMethods())
          .filter(m -> m.getName().equals("value"))
          .findFirst();
      if (method.isPresent()) {
        return method.get().invoke(enumValue);
      } else {
        return enumValue.name().toLowerCase();
      }
    } catch (IllegalAccessException e) {
      throw new IllegalArgumentException("Couldn't read value: " + enumValue, e);
    } catch (InvocationTargetException e) {
      throw new IllegalArgumentException("Couldn't get value: " + enumValue, e);
    }
  }

  /**
   * Get the list of values for an enum returned by invoking EnumType.getValues()
   *
   * @param m the fromValues method we're going to test.
   * @return a stream (because we're flatMapping) of enum values.
   * @throws IllegalArgumentException if we can't get the enum values for any reason.
   */
  @NotNull
  private static Stream<Enum<?>> getEnumValues(Method m) {
    final Class<?> declaringClass = m.getDeclaringClass();
    try {
      return Arrays.stream((Enum<?>[]) declaringClass.getDeclaredMethod("values").invoke(declaringClass));
    } catch (IllegalAccessException e) {
      throw new IllegalArgumentException("Couldn't read values: " + declaringClass.getSimpleName(), e);
    } catch (InvocationTargetException e) {
      throw new IllegalArgumentException("Couldn't get values: " + declaringClass.getSimpleName(), e);
    } catch (NoSuchMethodException e) {
      throw new IllegalArgumentException("Couldn't find 'values()' method: " + declaringClass.getSimpleName(), e);
    }
  }
}
