/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.config.dsl.config;

import static org.mule.runtime.ast.api.util.MuleAstUtils.createComponentParameterizationFromComponentAst;
import static org.mule.runtime.core.api.extension.provider.MuleExtensionModelProvider.getExtensionModel;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.test.allure.AllureConstants.SdkToolingSupport.SDK_TOOLING_SUPPORT;
import static org.mule.test.allure.AllureConstants.SdkToolingSupport.TopLevelElementsStory.TOP_LEVEL_ELEMENTS_CREATION;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.joining;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;

import org.mule.functional.junit4.AbstractArtifactAstTestCase;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.core.internal.context.MuleContextWithRegistry;
import org.mule.runtime.module.extension.api.runtime.config.ExtensionDesignTimeResolversFactory;
import org.mule.runtime.module.extension.internal.runtime.config.DefaultExtensionDesignTimeResolversFactory;
import org.mule.test.heisenberg.extension.HeisenbergExtension;
import org.mule.test.heisenberg.extension.model.KnockeableDoor;
import org.mule.test.heisenberg.extension.model.RecursivePojo;

import java.util.HashSet;
import java.util.Set;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature(SDK_TOOLING_SUPPORT)
@Story(TOP_LEVEL_ELEMENTS_CREATION)
public class TopLevelComponentCreationTestCase extends AbstractArtifactAstTestCase {

  private ExtensionDesignTimeResolversFactory extensionDesignTimeResolversFactory;

  @Before
  public void setUp() throws Exception {
    extensionDesignTimeResolversFactory = new DefaultExtensionDesignTimeResolversFactory();
    initialiseIfNeeded(extensionDesignTimeResolversFactory, muleContext);
  }

  @Test
  public void testTopLevelComponentCreation() throws Exception {
    // Test component references
    RecursivePojo referencedComponent = createTopLevelElement("referencedComponent", RecursivePojo.class);
    RecursivePojo mainComponent = createTopLevelElement("mainComponent", RecursivePojo.class);

    assertThat(mainComponent.getNext(), sameInstance(referencedComponent));

    // Test expressions
    KnockeableDoor doorComponent = createTopLevelElement("doorComponent", KnockeableDoor.class);
    assertThat(doorComponent.getAddress(), is("Calle Falsa 123"));
  }

  private <T> T createTopLevelElement(String componentName, Class<T> componentClass) throws Exception {
    ComponentAst componentAst = getComponent(componentName);

    ComponentParameterization<ParameterizedModel> parameterization =
        createComponentParameterizationFromComponentAst(componentAst);

    T topLevelComponent =
        extensionDesignTimeResolversFactory.createTopLevelComponent(parameterization, componentName, componentClass);

    ((MuleContextWithRegistry) muleContext).getRegistry().registerObject(componentName, topLevelComponent);

    return topLevelComponent;
  }

  private ComponentAst getComponent(String componentName) {
    return getAppAst().topLevelComponentsStream()
        .filter(componentAst -> componentAst.getComponentId().map(componentId -> componentId.equals(componentName)).orElse(false))
        .findFirst()
        .orElseThrow(() -> new IllegalStateException(format("Component '%s' not found among components '%s'", componentName,
                                                            getAppAst().topLevelComponentsStream()
                                                                .map(componentAst -> componentAst.getComponentId().orElse("null"))
                                                                .collect(joining(", ")))));
  }

  @Override
  protected String getConfigFile() {
    return "top-level-components-config.xml";
  }

  @Override
  protected Set<ExtensionModel> getRequiredExtensions() {
    return new HashSet(asList(getExtensionModel(), loadExtension(HeisenbergExtension.class, emptySet())));
  }
}
