/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.source;

import static org.mule.runtime.core.api.extension.provider.MuleExtensionModelProvider.getMuleVersion;

import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSet;
import org.mule.runtime.module.extension.internal.runtime.ResolverBasedComponentParameterization;
import org.mule.runtime.module.extension.internal.runtime.config.ResolverSetBasedParameterResolver;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;

import java.util.Optional;

/**
 * Implementation of {@link SourceFactoryContext} that delegates to {@link ResolverSet} for the {@link ComponentParameterization}.
 *
 * @since 4.10.0
 */
public final class ResolverBasedSourceFactoryContext implements SourceFactoryContext {

  private final ComponentParameterization<?> parameterization;
  private final ComponentLocation componentLocation;
  private final Optional<ConfigurationInstance> configurationInstance;
  private final Optional<ConnectionProvider<?>> connectionProvider;
  private final String defaultEncoding;
  private final MuleVersion runtimeVersion;

  /**
   * Creates a new instance
   *
   * @param sourceModel           the {@link SourceModel} describing the source
   * @param componentLocation     the {@link ComponentLocation} of the source component
   * @param sourceParameters      the {@link ResolverSet} containing the source parameters
   * @param configurationInstance the {@link ConfigurationInstance} if the source requires a configuration
   * @param connectionProvider    the {@link SourceConnectionProvider} if the source requires a connection
   * @param expressionManager     the {@link ExpressionManager} for expression evaluation
   * @param defaultEncoding       the artifact's default encoding
   */
  public ResolverBasedSourceFactoryContext(SourceModel sourceModel,
                                           ComponentLocation componentLocation,
                                           ResolverSet sourceParameters,
                                           Optional<ConfigurationInstance> configurationInstance,
                                           Optional<ConnectionProvider<?>> connectionProvider,
                                           ExpressionManager expressionManager,
                                           String defaultEncoding) {
    this.componentLocation = componentLocation;
    this.configurationInstance = configurationInstance;
    this.connectionProvider = connectionProvider;
    this.defaultEncoding = defaultEncoding;
    this.runtimeVersion = getMuleVersion();

    ParameterValueResolver parameterValueResolver = new ResolverSetBasedParameterResolver(sourceParameters,
                                                                                          sourceModel,
                                                                                          new ReflectionCache(),
                                                                                          expressionManager,
                                                                                          // For config overrides
                                                                                          configurationInstance);
    this.parameterization = new ResolverBasedComponentParameterization<>(sourceModel, parameterValueResolver);
  }

  @Override
  public ComponentParameterization<?> getParameterization() {
    return parameterization;
  }

  @Override
  public ComponentLocation getComponentLocation() {
    return componentLocation;
  }

  @Override
  public Optional<ConfigurationInstance> getConfigurationInstance() {
    return configurationInstance;
  }

  @Override
  public Optional<ConnectionProvider<?>> getConnectionProvider() {
    return connectionProvider;
  }

  @Override
  public String getDefaultEncoding() {
    return defaultEncoding;
  }

  @Override
  public MuleVersion getRuntimeVersion() {
    return runtimeVersion;
  }
}
